package ai.h2o.mojos.runtime.api.backend;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/** Access to directory-based fileset */
public class DirReaderBackend implements ReaderBackend {
    private static final Logger log = LoggerFactory.getLogger(DirReaderBackend.class);
    private final File dir;
    private final ResourceInfo.Cache cache = new ResourceInfo.Cache();

    private DirReaderBackend(File dir) {
        this.dir = dir;
    }

    public static ReaderBackend open(File dir) throws IOException {
        log.info("Opening mojo directory: {}", dir);
        if (!dir.isDirectory()) {
            throw new FileNotFoundException(dir.getAbsolutePath());
        }
        return new DirReaderBackend(dir.getCanonicalFile());
    }

    @Override
    public ResourceInfo getResourceInfo(final String resourceName) throws IOException {
        ResourceInfo info = cache.get(resourceName);
        if (info == null) {
            final File file = new File(dir, resourceName);
            if (! file.isFile()) {
                throw new FileNotFoundException(file.getAbsolutePath());
            }
            info = new ResourceInfo(file.length(), "FILESIZE:" + file.length()); // remember, it's considered weak anyway
        }
        return info;
    }

    @Override
    public InputStream getInputStream(String resourceName) throws FileNotFoundException {
        final File file = new File(dir, resourceName);
        if (! file.isFile()) {
            throw new FileNotFoundException(file.getAbsolutePath());
        }
        return new FileInputStream(file);
    }

    @Override
    public boolean exists(String resourceName) {
        final File file = new File(dir, resourceName);
        return file.isFile();
    }

    @Override
    public Collection<String> list() {
        final List<String> result = new ArrayList<>();
        listFiles(result, "", dir.listFiles());
        return result;
    }

    /**
     * Collects subtree file names, recursively, sorted by name
     */
    private void listFiles(List<String> result, String prefix, File[] files) {
        if (files == null) return;
        Arrays.sort(files);
        for (File file : files) {
            if (file.isFile()) {
                result.add(prefix + file.getName());
            } else if (!file.isDirectory()) {
                // we ignore strange file types, like devices pipes etc
                continue;
            }
            // recurse into subdirectories
            listFiles(result, prefix + file.getName() + "/", file.listFiles());
        }
    }

    @Override
    public void close() {}

    @Override
    public String toString() {
        return String.format("%s[%s]", getClass().getSimpleName(), dir);
    }
}
