This document describes how to use DAI's generated MOJO pipelines.

---

# About generated MOJO pipelines

DAI-generated MOJO pipelines are intended to be easily embeddable in any environment including JVM.
The only compilation and runtime dependency for a generated model is the MOJO runtime
file (e.g., `mojo2-runtime.jar`) produced as the build output of this package.

This overview covers following topics:

  * <a href="#whatisamojo">What is a MOJO?</a>
  * <a href="#quickstartmojo">MOJO Quick Start</a>

---

<a name="whatisamojo"></a>
# What is a MOJO?
A MOJO (Model Object, Optimized) is a binary format to represent DAI pipelines. The format support
easy deployment or embedding of pipelines without need of using heavy weight DAI runtime.

---

<a name="quickstartmojo"></a>
# MOJO Quick start

Each DAI experiment result can be exported as MOJO pipeline.

## Step 1: Start DAI, then build and extract the model
The examples below describe how to create a model using DAI UI. The finished
experiment shows option to build/download MOJO as a zip archive.
The downloaded archive contains the following files:

| Filename   | Description  |
|------------|--------------|
|`pipeline.mojo`     | The binary representation of pipeline in MOJO format.|
|`mojo2-runtime.jar` | A Java library to execute exported MOJO pipeline.   |
|`example.csv`       | A generated sample of data matching training data structure.  |
| `run_example.sh`   | A Linux shell script to transform example.csv data with exported MOJO pipeline. |
| `README.txt`       | A simple manual of using MOJO. |

## Step 2: Run the MOJO from Java

  1. Open a **new** terminal window and change current directory to a new `experiment` folder:

    ```bash
    $ mkdir experiment && cd experiment
    ```

  2. Create your main program in the `experiment` folder by creating a new file
     called `Main.java` (for example, using `vim Main.java`). Include the following contents.

    ```java
    import ai.h2o.mojos.runtime.MojoPipeline;
    import ai.h2o.mojos.runtime.frame.MojoFrame;
    import ai.h2o.mojos.runtime.frame.MojoFrameBuilder;
    import ai.h2o.mojos.runtime.frame.MojoRowBuilder;
    import ai.h2o.mojos.runtime.lic.LicenseException;
    import ai.h2o.mojos.runtime.utils.CsvWritingBatchHandler;
    import com.opencsv.CSVWriter;
    import java.io.BufferedWriter;
    import java.io.IOException;
    import java.io.OutputStreamWriter;
    import java.io.Writer;

    public class DocsExample {
        public static void main(String[] args) throws IOException, LicenseException {
            // Load model and csv
            final MojoPipeline model = MojoPipeline.loadFrom("pipeline.mojo");

            // Get and fill the input columns
            final MojoFrameBuilder frameBuilder = model.getInputFrameBuilder();
            final MojoRowBuilder rowBuilder = frameBuilder.getMojoRowBuilder();
            rowBuilder.setValue("AGE", "68");
            rowBuilder.setValue("RACE", "2");
            rowBuilder.setValue("DCAPS", "2");
            rowBuilder.setValue("VOL", "0");
            rowBuilder.setValue("GLEASON", "6");
            frameBuilder.addRow(rowBuilder);

            // Create a frame which can be transformed by MOJO pipeline
            final MojoFrame iframe = frameBuilder.toMojoFrame();

            // Transform input frame by MOJO pipeline
            final MojoFrame oframe = model.transform(iframe);
            // `MojoFrame.debug()` can be used to view the contents of a Frame
            // oframe.debug();

            // Output prediction as CSV
            final Writer writer = new BufferedWriter(new OutputStreamWriter(System.out));
            final CSVWriter csvWriter = new CSVWriter(writer);
            CsvWritingBatchHandler.csvWriteFrame(csvWriter, oframe, true);
        }
    }
    ```

  3. Compile the source code:
    ```bash
    $ javac -cp mojo2-runtime.jar Main.java
    ```

  4. Run the MOJO example:
    ```bash
    # Linux and OS X users
    $ java -cp .:mojo2-runtime.jar Main
    # Windows users
    $ java -cp .;mojo2-runtime.jar Main
    ```

  5. The following output is displayed:
    ```
	CAPSULE.True
	0.5442205910902282
	```
