package ai.h2o.mojos.runtime;

import ai.h2o.mojos.runtime.api.BasePipelineListener;
import ai.h2o.mojos.runtime.api.MojoPipelineService;
import ai.h2o.mojos.runtime.api.backend.ReaderBackend;
import ai.h2o.mojos.runtime.frame.MojoColumn;
import ai.h2o.mojos.runtime.frame.MojoFrame;
import ai.h2o.mojos.runtime.frame.MojoFrameBuilder;
import ai.h2o.mojos.runtime.frame.MojoFrameMeta;
import ai.h2o.mojos.runtime.lic.LicenseException;
import ai.h2o.mojos.runtime.readers.MojoReaderBackend;
import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import org.joda.time.DateTime;

/**
 * This class represents the MOJO scoring pipeline.
 * <p>
 * It transforms an instance of {@link MojoFrame} and produces a new {@link MojoFrame}.
 */
abstract public class MojoPipeline implements Serializable {

    private final String uuid;
    private final DateTime timestamp;
    private final String license;

    //----------------------------------------------------------------------------
    // Constructors
    //----------------------------------------------------------------------------

    protected MojoPipeline(String uuid, DateTime timestamp, String license) {
        this.uuid = uuid;
        this.timestamp = timestamp;
        this.license = license;
    }

    /**
     * Instantiate MojoPipeline from the provided .mojo file. This is the primary
     * method for creating MojoPipeline instances.
     *
     * @param file Name of the .mojo file with saved model information.
     * @return New MojoPipeline instance that is built according to the description
     * in the .mojo file.
     * @throws IOException In case the file cannot be read, or has invalid format,
     *                     or is not a Mojo-2 file.
     * @deprecated use {@link MojoPipelineService#loadPipeline(File)} instead
     */
    @Deprecated
    public static MojoPipeline loadFrom(String file) throws IOException, LicenseException {
        return MojoPipelineService.loadPipeline(new File(file));
    }

    /**
     * @deprecated use {@link MojoPipelineService#loadPipeline(ReaderBackend)} instead
     */
    @Deprecated
    public static MojoPipeline loadFrom(MojoReaderBackend mrb) throws IOException, LicenseException {
        return MojoPipelineService.loadPipeline(mrb.internalGetReaderBackend());
    }

    /**
     * Get the pipeline's uuid, if stated in the creation file.
     *
     * @return The model's uuid
     */
    public String getUuid() {
        return uuid;
    }

    /**
     * Get the creation time of the mojo file
     *
     * @return The mojo file's creation time
     */
    public DateTime getCreationTime() {
        return timestamp;
    }

    /**
     * Get the license of the mojo file
     *
     * @return the mojo file's license
     */
    public String getLicense() {
        return license;
    }

    /**
     * Get an instance of an {@link MojoFrameBuilder} that can be used to make an input frame
     *
     * @return A new input frame builder
     */
    public MojoFrameBuilder getInputFrameBuilder() {
        return getFrameBuilder(MojoColumn.Kind.Feature);
    }

    protected abstract MojoFrameBuilder getFrameBuilder(MojoColumn.Kind kind);

    /**
     * Get the meta data for the input frame (see {@link MojoFrameMeta}
     *
     * @return The input frame meta data
     */
    public MojoFrameMeta getInputMeta() {
        return getMeta(MojoColumn.Kind.Feature);
    }

    /**
     * Get the meta data for the output frame (see {@link MojoFrameMeta}
     *
     * @return The output frame meta data
     */
    public MojoFrameMeta getOutputMeta() {
        return getMeta(MojoColumn.Kind.Output);
    }

    protected abstract MojoFrameMeta getMeta(MojoColumn.Kind kind);

    /**
     * Executes the pipeline of transformers as stated in this model's mojo file.
     *
     * @param inputFrameBuilder A MojoFrameBuilder from which an input frame can be retrieved
     * @return A MojoFrame containing the results of the transform pipeline
     */
    public MojoFrame transform(MojoFrameBuilder inputFrameBuilder) {
        return transform(inputFrameBuilder.toMojoFrame());
    }

    /**
     * Executes the pipeline of transformers as stated in this model's mojo file.
     *
     * @param inputFrame A MojoFrame containing the input data
     * @return A MojoFrame containing the results of the transform pipeline
     * @todo mark this method `final` after https://github.com/h2oai/dai-deployment-templates/pull/172 is merged
     */
    public /*final*/ MojoFrame transform(MojoFrame inputFrame) {
        final MojoFrameMeta outputMeta = getMeta(MojoColumn.Kind.Output);
        final MojoFrame outputFrame = MojoFrameBuilder.getEmpty(outputMeta, inputFrame.getNrows());
        return transform(inputFrame, outputFrame);
    }

    /**
     * Executes the pipeline of transformers as stated in this model's mojo file.
     *
     * @param inputFrame  A MojoFrame containing the input data
     * @param outputFrame A MojoFrame to which the results of the transform pipeline should be written
     * @return `outputFrame` reference
     */
    public abstract MojoFrame transform(MojoFrame inputFrame, MojoFrame outputFrame);

    /**
     * Allows prediction contribution columns (computed by SHAP algo) to appear in the transformation result.
     * Current implementation removes the standard transformation result. This will be improved in future.
     *
     * @param enable - currently required to be true.
     */
    public void setShapPredictContrib(boolean enable) {
        if (enable) {
            throw new IllegalArgumentException("SHAP is not supported for this pipeline");
        }
    }

    public abstract void setListener(final BasePipelineListener listener);
}
