package ai.h2o.mojos.runtime.utils;

import org.joda.time.DateTime;
import org.joda.time.DateTimeFieldType;
import org.joda.time.DateTimeZone;
import org.joda.time.LocalDate;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatterBuilder;

import java.util.Objects;

/**
 * Date/Time holder, storing times in UTC timezone.
 */
public class MojoDateTime {
  // Default to fill missing date fields

  public static final DateTimeZone TIMEZONE = DateTimeZone.UTC;
  static private DateTime NOW = DateTime.now(TIMEZONE);

  public enum DatePart {
    YEAR("year") {
      @Override public String write(String var) { return "(" + var + ".getYear())"; }
    },
    QUARTER("quarter") {
      @Override public String write(String var) { return "(" + var + ".getQuarter())"; }
    },
    MONTH("month") {
      @Override public String write(String var) { return "(" + var + ".getMonth())"; }
    },
    WEEK("week") {
      @Override public String write(String var) { return "(" + var + ".getWeekOfYear())"; }
    },
    WEEKDAY("weekday") {
      @Override public String write(String var) { return  "(" + var + ".getDayOfWeek())"; }
    },
    DAY("day") {
      @Override public String write(String var) { return  "(" + var + ".getDay())"; }
    },
    DAYOFYEAR("dayofyear") {
      @Override public String write(String var) { return  "(" + var + ".getDayOfYear())"; }
    },
    HOUR("hour") {
      @Override public String write(String var) { return  "(" + var + ".getHour())"; }
    },
    MINUTE("minute") {
      @Override public String write(String var) { return  "(" + var + ".getMinute())"; }
    },
    SECOND("second") {
      @Override public String write(String var) { return  "(" + var + ".getSecond())"; }
    };

    private final String _name;

    DatePart(String name) { _name = name; }

    @Override public String toString() { return _name; }
    public abstract String write(String var);
  }

  private int _YEAR, _MONTH, _WEEK_OF_YEAR, _DAY, _DAY_OF_YEAR, _DAY_OF_WEEK, _HOUR, _MINUTE, _SECOND;
  private long _MILLIS_FROM_UNIX;

  public MojoDateTime(long millisFromUnix) {
    this(new DateTime(millisFromUnix, TIMEZONE));
  }

  private MojoDateTime(DateTime dt) {
    if (dt.isSupported(DateTimeFieldType.year())) {
      _YEAR = dt.get(DateTimeFieldType.year());
      if (dt.isSupported(DateTimeFieldType.monthOfYear())) {
        _MONTH = dt.get(DateTimeFieldType.monthOfYear());
        if (dt.isSupported(DateTimeFieldType.dayOfMonth())) {
          _DAY = dt.get(DateTimeFieldType.dayOfMonth());
        } else {
          _DAY = 1;
        }
      } else {
        _MONTH = 1;
        _DAY = 1;
      }
    } else {
      _YEAR = NOW.getYear();
      _MONTH = NOW.getMonthOfYear();
      _DAY = NOW.getDayOfMonth();
    }
    LocalDate tmp = new LocalDate(_YEAR, _MONTH, _DAY);

    if (dt.isSupported(DateTimeFieldType.dayOfYear()) && dt.isSupported(DateTimeFieldType.dayOfWeek())) {
      _DAY_OF_YEAR = dt.get(DateTimeFieldType.dayOfYear());
      _DAY_OF_WEEK = dt.get(DateTimeFieldType.dayOfWeek());
    } else {
      _DAY_OF_YEAR = tmp.getDayOfYear();
      _DAY_OF_WEEK = tmp.getDayOfWeek();
    }
    if (dt.isSupported(DateTimeFieldType.weekOfWeekyear())) {
      _WEEK_OF_YEAR = dt.get(DateTimeFieldType.weekOfWeekyear());
    }

    if (dt.isSupported(DateTimeFieldType.hourOfDay())) {  // FIXME: what about halfHourOfDay
      _HOUR = dt.get(DateTimeFieldType.hourOfDay());
      if (dt.isSupported(DateTimeFieldType.minuteOfHour())) {
        _MINUTE = dt.get(DateTimeFieldType.minuteOfHour());
        if (dt.isSupported(DateTimeFieldType.secondOfMinute())) {
          _SECOND = dt.get(DateTimeFieldType.secondOfMinute());
        } else {
          _SECOND = 0;
        }
      } else {
        _MINUTE = 0;
        _SECOND = 0;
      }
    } else {
      _HOUR = 0;
      _MINUTE = 0;
      _SECOND = 0;
    }
    _MILLIS_FROM_UNIX = dt.getMillis();
  }

  public static MojoDateTime parse(String s) {
    DateTime out = DateParser.DTF.withZone(TIMEZONE).parseDateTime(s);
    return new MojoDateTime(out);
  }

  public static MojoDateTime parse(String s, DateParser parser) {
    return parser.parse(s);
  }

  public static MojoDateTime create(DateTime s) {
    return new MojoDateTime(s);
  }

  public int getYear() {
    return _YEAR;
  }

  public int getQuarter() {
    return (_MONTH - 1) / 3 + 1;
  }

  public int getMonth() {
    return _MONTH;
  }

  public int getDay() {
    return _DAY;
  }

  public int getDayOfYear() {
    return _DAY_OF_YEAR;
  }

  public int getWeekOfYear() {
    return _WEEK_OF_YEAR;
  }

  public int getDayOfWeek() {
    return _DAY_OF_WEEK - 1;
  }

  public int getHour() {
    return _HOUR;
  }

  public int getMinute() {
    return _MINUTE;
  }

  public int getSecond() {
    return _SECOND;
  }

  public long getMillis() {
    return _MILLIS_FROM_UNIX;
  }

  public static final DateTimeFormatterBuilder genFormatterBuilder(DateTimeFormatterBuilder dtfb, String[] formats) {
    for (String f : formats) {
      dtfb.appendOptional(DateTimeFormat.forPattern(f).getParser());
    }
    return dtfb;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }
    MojoDateTime that = (MojoDateTime) o;
    return _YEAR == that._YEAR &&
           _MONTH == that._MONTH &&
           _WEEK_OF_YEAR == that._WEEK_OF_YEAR &&
           _DAY == that._DAY &&
           _DAY_OF_YEAR == that._DAY_OF_YEAR &&
           _DAY_OF_WEEK == that._DAY_OF_WEEK &&
           _HOUR == that._HOUR &&
           _MINUTE == that._MINUTE &&
           _SECOND == that._SECOND;
  }

  @Override
  public int hashCode() {

    return Objects
        .hash(_YEAR, _MONTH, _WEEK_OF_YEAR, _DAY, _DAY_OF_YEAR, _DAY_OF_WEEK, _HOUR, _MINUTE,
              _SECOND);
  }

  @Override
  public String toString() {
    return "MojoDateTime{" +
           "_YEAR=" + _YEAR +
           ", _MONTH=" + _MONTH +
           ", _WEEK_OF_YEAR=" + _WEEK_OF_YEAR +
           ", _DAY=" + _DAY +
           ", _DAY_OF_YEAR=" + _DAY_OF_YEAR +
           ", _DAY_OF_WEEK=" + _DAY_OF_WEEK +
           ", _HOUR=" + _HOUR +
           ", _MINUTE=" + _MINUTE +
           ", _SECOND=" + _SECOND +
           '}';
  }


}
