package app.valuationcontrol.multimodule.library.entities;

import app.valuationcontrol.multimodule.library.helpers.DataTransformer;
import app.valuationcontrol.multimodule.library.records.ModelData;
import app.valuationcontrol.multimodule.library.records.SingleModelAccessData;
import app.valuationcontrol.multimodule.library.records.UserData;
import jakarta.persistence.Column;
import jakarta.persistence.ElementCollection;
import jakarta.persistence.Entity;
import jakarta.persistence.GeneratedValue;
import jakarta.persistence.GenerationType;
import jakarta.persistence.Id;
import jakarta.persistence.Table;
import jakarta.persistence.UniqueConstraint;
import java.util.*;
import lombok.Getter;
import org.springframework.security.access.AccessDeniedException;

/**
 * This class is an entity that a variable within a model
 *
 * @author thomas
 */
@Getter
@Entity
@Table(
    name = "\"User\"",
    uniqueConstraints = {@UniqueConstraint(columnNames = {"user_email"})})
public class User implements DataTransformer<UserData> {

  public enum MODEL_ROLE {
    ADMIN,
    EDITOR,
    READER
  }

  @Id
  @GeneratedValue(strategy = GenerationType.IDENTITY)
  private long id;

  @Column(name = "user_email")
  private String email;

  @Column(name = "user_sub")
  private String sub;

  @ElementCollection private final Map<Model, String> modelRoles = new HashMap<>();

  public void setEmail(String email) {
    this.email = email;
  }

  public String getModelRoleWithModelId(long modelId) {
    return modelRoles.entrySet().stream()
        .filter(entry -> entry.getKey().getId() == modelId)
        .findFirst()
        .map(Map.Entry::getValue)
        .orElse(null);
  }

  public boolean getModelIsLocked(long modelId) {
    return modelRoles.entrySet().stream()
        .anyMatch(entry -> entry.getKey().getId() == modelId && entry.getKey().isLocked());
  }

  /**
   * Check if the user has sufficient privilege to run the action
   *
   * @param requiredRole - the requiredRole
   * @param modelId - the id of the model to check access for
   * @param isAnUnlockRequest indicates wether this is a specific unlock request
   * @return true if yes, false if not
   */
  public boolean validateRole(String requiredRole, Long modelId, boolean isAnUnlockRequest)
      throws AccessDeniedException {

    String userRole = getModelRoleWithModelId(modelId);
    boolean isLocked = getModelIsLocked(modelId);
    // Returning false if the user does not have access to the model at all
    if (userRole == null) {
      throw new AccessDeniedException("User does not have access to this model");
    }

    final MODEL_ROLE userModelRole = MODEL_ROLE.valueOf(userRole);
    final MODEL_ROLE requiredModelRole = MODEL_ROLE.valueOf(requiredRole);

    // if an unlock request, bypass the other test and check if the user has the required right
    // (ADMIN(
    if (isAnUnlockRequest && userModelRole.equals(requiredModelRole)) {
      return true;
    }

    // If requested query requires reader then grant read, else return false
    if (isLocked && MODEL_ROLE.READER.equals(requiredModelRole)) {
      return true;
    }

    if (isLocked
        && (MODEL_ROLE.EDITOR.equals(requiredModelRole)
            || MODEL_ROLE.ADMIN.equals(requiredModelRole))) {
      throw new AccessDeniedException(
          "Model is locked for changes. Ask an administrator to unlock the model.");
    }

    if (userModelRole.equals(MODEL_ROLE.ADMIN)) {
      return true;
    }

    if (MODEL_ROLE.EDITOR.equals(userModelRole)
        && (MODEL_ROLE.EDITOR.equals(requiredModelRole)
            || MODEL_ROLE.READER.equals(requiredModelRole))) {
      return true;
    }

    return MODEL_ROLE.READER.equals(userModelRole) && MODEL_ROLE.READER.equals(requiredModelRole);
  }

  public void addModel(Model model, MODEL_ROLE role) {
    this.getModelRoles().put(model, role.name());
  }

  @Override
  public UserData asData() {
    final List<SingleModelAccessData> asData = new ArrayList<>();
    modelRoles.forEach(
        (model, access) -> {
          SingleModelAccessData accessData =
              new SingleModelAccessData(MODEL_ROLE.valueOf(access), model.asData());
          asData.add(accessData);
        });
    asData.sort(
        Comparator.comparing(
            SingleModelAccessData::modelData, Comparator.comparing(ModelData::name)));
    return new UserData(this.getId(), this.getEmail(), this.getSub(), asData);
  }
}
