package app.valuationcontrol.multimodule.library.helpers;

import app.valuationcontrol.multimodule.library.entities.Event;
import app.valuationcontrol.multimodule.library.entities.Variable;
import java.util.List;
import java.util.Locale;
import lombok.extern.log4j.Log4j2;
import org.springframework.context.event.EventListener;
import org.springframework.data.util.Pair;
import org.springframework.stereotype.Component;

@Component
@Log4j2
public class VariableNameChangeListener {

  private static final String WORD_BOUNDARY = "\\b";
  public static final String ZERO_OR_MORE_CHARACTERS = ".*";
  private final EntityService entityService;

  public VariableNameChangeListener(EntityService entityService) {
    this.entityService = entityService;
  }

  @EventListener
  private void handleEvent(Event<Variable> variableEvent) {
    if (isNameChanged(variableEvent)) {
      Variable myNewVariable = variableEvent.getNewVersion();
      List<Variable> dependants = getDependants(myNewVariable);

      final List<Pair<Variable, Variable>> pairs =
          dependants.stream()
              .map(
                  variable -> {
                    Variable oldVariable = new Variable(variable);

                    this.updateFormula(variable);

                    return Pair.of(oldVariable, variable);
                  })
              .toList();

      entityService.updateAll(Variable.class, pairs);
    }

    // When a variable is created afterwards we need to keep variable dependencies updated
    if (variableEvent.isCreate()) {

      final Variable createdVariable = variableEvent.getNewVersion();

      // Temporarily add the variable to the model to get the new evaluated formulas working
      createdVariable.getAttachedModel().getVariables().add(createdVariable);

      try {
        var variableName =
            contains(exactWord(createdVariable.getVariableName().toLowerCase(Locale.ROOT)));

        final List<Pair<Variable, Variable>> variablesDependingOnCreatedVariable =
            createdVariable.getAttachedModel().getVariables().stream()
                .filter(variable -> variable.getEvaluatedVariableFormula() != null)
                .filter(variable -> variable.getEvaluatedVariableFormula().matches(variableName))
                .map(
                    variable -> {
                      Variable oldVariable = new Variable(variable);

                      variable.setEvaluatedVariableFormula(
                          FormulaEvaluator.evaluateVariableFormula(variable));
                      return Pair.of(oldVariable, variable);
                    })
                .toList();

        if (!variablesDependingOnCreatedVariable.isEmpty()) {
          entityService.updateAll(Variable.class, variablesDependingOnCreatedVariable);
        }
      } finally {
        createdVariable.getAttachedModel().getVariables().remove(createdVariable);
      }
    }
  }

  /**
   * Rebuilds the formula based on its dependencies
   *
   * @param variable the variable to change formula in
   */
  private void updateFormula(Variable variable) {
    if (variable.getEvaluatedVariableFormula() != null) {
      String tempFormula = variable.getEvaluatedVariableFormula();
      for (Variable variableDependency : variable.getVariableDependencies()) {
        tempFormula =
            tempFormula.replaceAll(
                "\\{" + variableDependency.getId() + "\\}", variableDependency.getVariableName());
      }
      variable.setVariableFormula(tempFormula);
    }
  }

  /**
   * Produce a regex replacing the exact word
   *
   * @param variableName the word
   * @return regex for the word sent in.
   */
  public static String exactWord(String variableName) {
    return WORD_BOUNDARY + variableName + WORD_BOUNDARY;
  }

  /** produce a regex producing a contains regexp */
  public static String contains(String value) {
    return ZERO_OR_MORE_CHARACTERS + value + ZERO_OR_MORE_CHARACTERS;
  }

  /**
   * Find which variables depends on this
   *
   * @param variable the variable to find dependants for
   * @return a list of variables depending on this variable
   */
  private List<Variable> getDependants(Variable variable) {
    return variable.getAttachedModel().getVariables().stream()
        .filter(
            otherVariable ->
                otherVariable.getVariableDependencies().stream()
                    .mapToLong(Variable::getId)
                    .anyMatch(value -> value == variable.getId()))
        .toList();
  }

  /**
   * Checks if name is changed
   *
   * @param variableEvent the event
   * @return true if the name is changed.
   */
  private boolean isNameChanged(Event<Variable> variableEvent) {
    return variableEvent.isUpdate()
        && !variableEvent
            .getOldVersion()
            .getVariableName()
            .equals(variableEvent.getNewVersion().getVariableName());
  }
}
