package app.valuationcontrol.multimodule.library.helpers.stompInterceptors;

import app.valuationcontrol.multimodule.library.helpers.exceptions.ResourceException;
import app.valuationcontrol.multimodule.library.helpers.security.MyUserPrincipal;
import app.valuationcontrol.multimodule.library.xlhandler.SCENARIO;
import java.security.Principal;
import java.util.Arrays;
import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import lombok.extern.log4j.Log4j2;
import org.springframework.http.HttpStatus;
import org.springframework.lang.NonNull;
import org.springframework.messaging.Message;
import org.springframework.messaging.MessageChannel;
import org.springframework.messaging.simp.stomp.StompCommand;
import org.springframework.messaging.simp.stomp.StompHeaderAccessor;
import org.springframework.messaging.support.ChannelInterceptor;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;

@Log4j2
public class TopicSubscriptionInterceptor implements ChannelInterceptor {

  public static final String TOPIC_MODEL = "/topic/model/";
  public final String TOPIC_CHANGELOG = "/topic/changelog/";

  final String REGEX =
      "(\\/topic\\/model\\/|\\/topic\\/changelog\\/)([0-9]*)((\\/)([0-9]*))?"; // Last regex allow
  // all figures for
  // scenarioNumber so
  // that we test below

  final String[] topicsRequiringValidation = {TOPIC_MODEL, TOPIC_CHANGELOG};
  final String[] otherTopics = {"/user/topic/errors"};

  @Override
  public Message<?> preSend(@NonNull Message<?> message, @NonNull MessageChannel channel) {

    StompHeaderAccessor headerAccessor = StompHeaderAccessor.wrap(message);
    if (StompCommand.SUBSCRIBE.equals(headerAccessor.getCommand())) {
      log.debug(
          "Checking " + headerAccessor.getCommand() + " -> " + headerAccessor.getDestination());
      if (Arrays.stream(topicsRequiringValidation)
          .anyMatch(
              topic -> Objects.requireNonNull(headerAccessor.getDestination()).contains(topic))) {
        Principal userPrincipal = headerAccessor.getUser();
        log.debug(userPrincipal);
        try {
          assert userPrincipal != null;
          MyUserPrincipal mp =
              (MyUserPrincipal)
                  ((UsernamePasswordAuthenticationToken) userPrincipal).getPrincipal();
          log.debug(mp.getUsername());

          Pattern pattern = Pattern.compile(REGEX);
          Matcher matcher = pattern.matcher(headerAccessor.getDestination());
          long modelId = 0;

          if (matcher.find()) { // Expect only once
            modelId = Long.parseLong(matcher.group(2));
            if (matcher.groupCount() > 3 && matcher.group(3) != null) {
              int scenarioNumber = Integer.parseInt(matcher.group(5));
              SCENARIO.from(
                  scenarioNumber); // This should result in an exception if trying to access an
              // erronoeus scenario
            }
          }

          if (modelId > 0 && mp.hasModelRole(modelId, "READER")) {
            log.debug("Successful subscription");
          } else {
            throw new IllegalArgumentException("No permission for this topic");
          }
        } catch (NumberFormatException e) {
          log.debug(e);
          throw new ResourceException(
              HttpStatus.BAD_REQUEST, "Unable to find the model to subscribe to");
        } catch (Exception e) {
          log.debug(e);
          throw new IllegalArgumentException("No permission for this topic");
        }
      } else if (Arrays.stream(otherTopics)
          .noneMatch(
              topic -> Objects.requireNonNull(headerAccessor.getDestination()).contains(topic))) {
        log.debug(
            "Requested subscription "
                + headerAccessor.getDestination()
                + " is not recognized by the topic interceptor");
        throw new IllegalArgumentException("Requested topic does not exits");
      }
    }
    return message;
  }
}
