package cc.redberry.rings.poly;

import cc.redberry.rings.poly.univar.IUnivariatePolynomial;
import cc.redberry.rings.poly.univar.IrreduciblePolynomials;
import cc.redberry.rings.poly.univar.UnivariateDivision;
import cc.redberry.rings.poly.univar.UnivariateGCD;

import java.util.Iterator;

/**
 * Algebraic number field {@code F(α)} represented as a simple field extension, for details see {@link
 * SimpleFieldExtension}.
 *
 * @see SimpleFieldExtension
 * @see FiniteField
 * @see cc.redberry.rings.Rings#AlgebraicNumberField(IUnivariatePolynomial)
 * @since 2.5
 */
public class AlgebraicNumberField<E extends IUnivariatePolynomial<E>>
        extends SimpleFieldExtension<E> {

    /**
     * Constructs a simple field extension {@code F(α)} generated by the algebraic number {@code α} with the specified
     * minimal polynomial.
     *
     * <p><b>NOTE:</b> irreducibility test for the minimal polynomial is not performed here, use {@link
     * IrreduciblePolynomials#irreducibleQ(IUnivariatePolynomial)} to test irreducibility.
     *
     * @param minimalPoly the minimal polynomial
     */
    public AlgebraicNumberField(E minimalPoly) {
        super(minimalPoly);
        if (minimalPoly.isOverFiniteField())
            throw new IllegalArgumentException("Use FiniteField for constructing extensions of finite fields.");
    }

    @Override
    public boolean isField() {return minimalPoly.isOverField();}

    @Override
    public boolean isUnit(E element) {
        return (isField() && !isZero(element)) || (isOne(element) || isMinusOne(element));
    }

    @Override
    public E gcd(E a, E b) {
        // NOTE: don't change this
        return isField() ? a : UnivariateGCD.PolynomialGCD(a, b);
    }

    @Override
    public E[] divideAndRemainder(E a, E b) {
        if (isField())
            return createArray(multiply(a, reciprocal(b)), getZero());

        // the following code has no any particular math meaning,
        // just to make some programming things easier

        E[] cancel = normalizer2(b);
        E cb = cancel[0];
        E bcb = cancel[1];

        E pseudoQuot = multiply(a, cb);
        E quot = pseudoQuot.clone().divideByLC(bcb);
        if (quot != null)
            return createArray(quot, getZero());

        return UnivariateDivision.divideAndRemainder(a, b, true);
    }

    @Override
    public E remainder(E dividend, E divider) {
        return divideAndRemainder(dividend, divider)[1];
    }

    /**
     * Gives an element {@code C(element)} of this field extension with the property that {@code element * C(element)}
     * is in the base field.
     */
    public E normalizer(E element) {
        return normalizer2(element)[0];
    }

    E[] normalizer2(E element) {
        if (isField())
            return createArray(reciprocal(element), getOne());

        if (element.isZero())
            throw new ArithmeticException("divide by zero");
        if (isOne(element))
            return createArray(element, getOne());
        if (isMinusOne(element))
            return createArray(element, getOne());

        E[] xgcd = UnivariateGCD.PolynomialExtendedGCD(element, minimalPoly);
        E conjugate = xgcd[1];
        E content = conjugate.contentAsPoly();
        return createArray(conjugate, xgcd[0].divideByLC(content));
    }

    @Override
    public Iterator<E> iterator() {
        throw new UnsupportedOperationException("this field has infinite cardinality");
    }
}
