/*
 * Id$: zuv-cloud:z-service:cc.zuv.service.aliyun.mts.AliMtsService:20181230232035
 *
 * AliMtsService.java
 * Copyright (c) 2002-2020 Luther Inc.
 * http://zuv.cc
 * All rights reserved.
 */

package cc.zuv.service.aliyun.mts;

import cc.zuv.ZuvException;
import cc.zuv.document.support.json.GsonParser;
import cc.zuv.lang.StringUtils;
import cc.zuv.service.aliyun.mts.message.*;
import cc.zuv.utility.CodecUtils;
import com.aliyuncs.DefaultAcsClient;
import com.aliyuncs.IAcsClient;
import com.aliyuncs.exceptions.ClientException;
import com.aliyuncs.mts.model.v20140618.*;
import com.aliyuncs.profile.DefaultProfile;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;

/**
 * zuv-cloud File Description
 *
 * @author          Kama Luther
 * @version         0.1
 * @since           0.1
 * @create.date     2018-12-30 23:20
 * @modify.date     2018-12-30 23:20
 */
@Slf4j
@Component
public class AliMtsService
{

    //-----------------------------------------------------------------------------------------

    private final String TranscodeTimeoutCancelCode = "TranscodeTimeoutCancel";

    //-----------------------------------------------------------------------------------------

    @Autowired
    private AliMtsConfig aliMtsConfig;

    //-----------------------------------------------------------------------------------------

    private String getInterHost(String host)
    {
        return host + MTS_HOST_INTERNET;
    }
    private String getIntraHost(String host)
    {
        return host + MTS_HOST_INTRANET;
    }


    //MTS内网域名: ://mts.cn-beijing-internal.aliyuncs.com
    private String getMtsInterEndpoint(String scheme, String host)
    {
        return scheme + "://" + MTS_HOST_PREFIX + getInterHost(host);
    }
    //MTS外网域名: ://mts.cn-beijing.aliyuncs.com
    private String getMtsIntraEndpoint(String scheme, String host)
    {
        return scheme + "://" + MTS_HOST_PREFIX + getIntraHost(host);
    }

    public String getMtsEndpoint()
    {
        String  scheme      = aliMtsConfig.isUsehttps()?HTTPS:HTTP;
        boolean useinter    = aliMtsConfig.isUseinter();
        String  host        = aliMtsConfig.getRegion();
        return useinter?getMtsInterEndpoint(scheme, host):getMtsIntraEndpoint(scheme, host);
    }

    private String getOssRegion(String region)
    {
        return OSS_HOST_PREFIX + region;
    }

    //-----------------------------------------------------------------------------------------

    private static final String HTTP    = "http";
    private static final String HTTPS   = "https";

    //-----------------------------------------------------------------------------------------

    //cn-hangzhou、cn-shenzhen、cn-shanghai、cn-beijing
    private static final String MTS_HOST_PREFIX = "mts.";
    private static final String OSS_HOST_PREFIX = "oss-";
    private static final String MTS_HOST_INTRANET = ".aliyuncs.com";
    private static final String MTS_HOST_INTERNET = "-internal.aliyuncs.com";

    //-----------------------------------------------------------------------------------------

    private IAcsClient client;

    public void initial()
    {
        if(aliMtsConfig==null)
        {
            throw new ZuvException("初始配置失败");
        }

        //
        String  region      = aliMtsConfig.getRegion();
        String  key         = aliMtsConfig.getAccount().getKey();
        String  secret      = aliMtsConfig.getAccount().getSecret();

        // 创建DefaultAcsClient实例并初始化
        DefaultProfile profile = DefaultProfile.getProfile(region, key, secret);
        client = new DefaultAcsClient(profile);
    }

    //-----------------------------------------------------------------------------------------

    // output_{Count}.jpg
    public MtsSubmitJobRes submitSnapshotJob(String sourcebucket, String sourcefilepath,
                                             String targetbucket, String targetfilepath)
    {
        //
        String  region      = aliMtsConfig.getRegion();
        String  ossregion   = getOssRegion(region);

        //源
        String encsourcefilepath = CodecUtils.url_encode(sourcefilepath);
        String input = GsonParser.json(new OssFile(ossregion, sourcebucket, encsourcefilepath));

        //目标
        String enctargetfilepath = CodecUtils.url_encode(targetfilepath);
        OssSnapshotConfig config = new OssSnapshotConfig();
        config.setOutputFile(new OssFile(ossregion, targetbucket, enctargetfilepath));
        config.setTime("10"); //截图开始时间 毫秒
        config.setInterval("5"); //截图间隔时间 秒
        config.setNum("5"); //截图数量
        config.setHeight("360");
        String snapshotconfig = GsonParser.json(config);

        // 创建API请求并设置参数
        SubmitSnapshotJobRequest request = new SubmitSnapshotJobRequest();
        request.setInput(input);
        request.setSnapshotConfig(snapshotconfig);

        //
        String pipelineid = aliMtsConfig.getPipeline().getId();
        request.setPipelineId(pipelineid);

        // 发起请求并处理应答或异常
        SubmitSnapshotJobResponse response;
        try
        {
            response = client.getAcsResponse(request);

            MtsSubmitJobRes res = new MtsSubmitJobRes();
            res.setRequestId(response.getRequestId());
            res.setJobId(response.getSnapshotJob().getId());
            log.info("RequestId is:" + res.getRequestId());
            log.info("JobId is:" + res.getJobId());
            return res;
        }
        catch (ClientException e)
        {
            log.error("初始配置失败 {}", e.getMessage());
            throw new ZuvException("初始配置失败", e);
        }
    }

    public MtsSubmitJobRes submitTranscodeJob(String sourcebucket, String sourcefilepath,
                                   String targetbucket, String targetfilepath)
    {
        //
        AliMtsConfig.Transcode transcodecfg = aliMtsConfig.getTranscode();

        //
        String  region      = aliMtsConfig.getRegion();
        String  ossregion   = getOssRegion(region);

        //源
        String encsourcefilepath = CodecUtils.url_encode(sourcefilepath);
        String input = GsonParser.json(new OssFile(ossregion, sourcebucket, encsourcefilepath));

        //目标
        String enctargetfilepath = CodecUtils.url_encode(targetfilepath);
        OssTranscodeOutput outputobj = new OssTranscodeOutput();
        String transcodetplid = transcodecfg.getTplid();
        if(StringUtils.NotEmpty(transcodetplid)) outputobj.setTemplateId(transcodetplid);
        outputobj.setOutputObject(enctargetfilepath);

        String type = transcodecfg.getType()!=null?transcodecfg.getType():"mp4";
        String vcodec = transcodecfg.getVideoCodec()!=null?transcodecfg.getVideoCodec():"H.264";
        String vbitrate = transcodecfg.getVideoBitrate();
        String vwidth = transcodecfg.getVideoWidth();
        String vfps = transcodecfg.getVideoFps();
        String acodec = transcodecfg.getAudioCodec()!=null?transcodecfg.getAudioCodec():"AAC";
        String abitrate = transcodecfg.getAudioBitrate();
        outputobj.setContainer(new OssTranscodeOutput.Container(type));
        outputobj.setVideo(new OssTranscodeOutput.Video(vcodec, vbitrate, vwidth, vfps));
        outputobj.setAudio(new OssTranscodeOutput.Audio(acodec, abitrate, "2", "44100"));

        OssTranscodeOutput[] outputs = new OssTranscodeOutput[]{outputobj};
        String output = GsonParser.json(outputs);

        // 创建API请求并设置参数
        SubmitJobsRequest request = new SubmitJobsRequest();
        request.setInput(input);
        request.setOutputs(output);
        request.setOutputLocation(ossregion);
        request.setOutputBucket(targetbucket);

        //
        String pipelineid = aliMtsConfig.getPipeline().getId();
        request.setPipelineId(pipelineid);

        // 发起请求并处理应答或异常
        long timeout = transcodecfg.getTimeout(); //秒
        SubmitJobsResponse response;
        try
        {
            response = client.getAcsResponse(request);

            MtsSubmitJobRes res = new MtsSubmitJobRes();
            res.setRequestId(response.getRequestId());
            log.info("RequestId is:" + res.getRequestId());
            res.setSuccess(response.getJobResultList().get(0).getSuccess());
            if(res.isSuccess())
            {
                res.setJobId(response.getJobResultList().get(0).getJob().getJobId());
                log.info("SubmitJobs Success, JobId is:" + res.getJobId());

                //查询状态
                int count = 0;
                while (true)
                {
                    MtsQueryJobRes queryres = submitQueryJob(res.getJobId());
                    if(queryres.getCode()!=null)
                    {
                        log.error("QueryJobs Failure. state: {} percent:{}", queryres.getCode(), queryres.getMessage());
                        break;
                    }
                    else if("TranscodeSuccess".equalsIgnoreCase(queryres.getState())
                        || 100 == queryres.getPercent())
                    {
                        log.info("QueryJobs Success.");
                        break;
                    }
                    else
                    {
                        log.debug("QueryJobs state: {} percent:{}", queryres.getState(), queryres.getPercent());
                    }

                    try { Thread.sleep(1000L); } catch (InterruptedException ignored) { }

                    //
                    if(++count>timeout)
                    {
                        queryres.setCode(TranscodeTimeoutCancelCode);
                        queryres.setMessage("转码超时取消操作: " + count);
                        log.error("SubmitJobs Failure, code:" + queryres.getCode() + " message:" + queryres.getMessage());
                        break;
                    }
                }
            }
            else
            {
                res.setCode(response.getJobResultList().get(0).getCode());
                res.setMessage(response.getJobResultList().get(0).getMessage());
                log.error("SubmitJobs Failure, code:" + res.getCode() + " message:" + res.getMessage());
            }

            return res;
        }
        catch (ClientException e)
        {
            log.error("初始配置失败 {}", e.getMessage());
            throw new ZuvException("初始配置失败", e);
        }
    }

    public MtsSubmitJobRes submitWatermarkJob(String sourcebucket, String sourcefilepath,
                                   String markbucket, String markfilepath,
                                   String content,
                                   String targetbucket, String targetfilepath)
    {
        //
        AliMtsConfig.Watermark watermarkcfg = aliMtsConfig.getWatermark();

        //
        String  region      = aliMtsConfig.getRegion();
        String  ossregion   = getOssRegion(region);

        //源
        String encsourcefilepath = CodecUtils.url_encode(sourcefilepath);
        String input = GsonParser.json(new OssFile(ossregion, sourcebucket, encsourcefilepath));

        //目标
        String enctargetfilepath = CodecUtils.url_encode(targetfilepath);
        OssWatermarkOutput outputobj = new OssWatermarkOutput();
        String watermarktplid = watermarkcfg.getTplid();
//        outputobj.setTemplateId(watermarktplid);
        outputobj.setOutputObject(enctargetfilepath);

        List<OssWatermarkOutput.WaterMark> watermarks = new ArrayList<>();

        //Image
        String imageWidth = watermarkcfg.getWidth();
        String imageReferpos = watermarkcfg.getReferPos();
        String imagedx = watermarkcfg.getDx();
        String imagedy = watermarkcfg.getDy();
        OssWatermarkOutput.WaterMark imagewm = new OssWatermarkOutput.WaterMark();
        imagewm.setType("Image");
        imagewm.setReferPos(imageReferpos); // ReferPos.TopRight.name()
        imagewm.setWidth(imageWidth); //20,0.05
        imagewm.setDx(imagedx); //0 10
        imagewm.setDy(imagedy); //0 10
        imagewm.setWaterMarkTemplateId(watermarktplid);
        imagewm.setInputFile(new OssFile(ossregion, markbucket, markfilepath));
        watermarks.add(imagewm);

        //Text
        if(StringUtils.NotEmpty(content))
        {
            String fontsize = watermarkcfg.getFontSize();
            String fontcolor = watermarkcfg.getFontColor();
            String fontalpha = watermarkcfg.getFontAlpha();
            String fonttop = watermarkcfg.getFontLeft();
            String fontleft = watermarkcfg.getFontLeft();
            OssWatermarkOutput.TextWaterMark text = new OssWatermarkOutput.TextWaterMark();
            text.setContent(CodecUtils.base64_encode(content));
            text.setFontName("SimSun");
            text.setFontSize(fontsize); //16
            text.setFontColor(fontcolor); //Red
            text.setFontAlpha(fontalpha); //0.5
            text.setTop(fonttop); //10
            text.setLeft(fontleft); //10

            OssWatermarkOutput.WaterMark textwm = new OssWatermarkOutput.WaterMark();
            textwm.setType("Text");
            textwm.setWaterMarkTemplateId(watermarktplid);
            textwm.setTextWaterMark(text);
            watermarks.add(textwm);
        }

        outputobj.setWaterMarks(watermarks);
        OssWatermarkOutput[] outputs = new OssWatermarkOutput[]{outputobj};
        String output = GsonParser.json(outputs);

        // 创建API请求并设置参数
        SubmitJobsRequest request = new SubmitJobsRequest();
        request.setInput(input);
        request.setOutputs(output);
        request.setOutputLocation(ossregion);
        request.setOutputBucket(targetbucket);

        //
        String pipelineid = aliMtsConfig.getPipeline().getId();
        request.setPipelineId(pipelineid);

        // 发起请求并处理应答或异常
        SubmitJobsResponse response;
        long timeout = watermarkcfg.getTimeout(); //秒
        try
        {
            response = client.getAcsResponse(request);

            MtsSubmitJobRes res = new MtsSubmitJobRes();
            res.setRequestId(response.getRequestId());
            log.info("RequestId is:" + res.getRequestId());
            res.setSuccess(response.getJobResultList().get(0).getSuccess());
            if(res.isSuccess())
            {
                res.setJobId(response.getJobResultList().get(0).getJob().getJobId());
                log.info("SubmitJobs Success, JobId is:" + res.getJobId());

                //查询状态
                int count = 0;
                while (true)
                {
                    MtsQueryJobRes queryres = submitQueryJob(res.getJobId());
                    if(queryres.getCode()!=null)
                    {
                        log.error("QueryJobs Failure. state: {} percent:{}", queryres.getCode(), queryres.getMessage());
                        break;
                    }
                    else if("TranscodeSuccess".equalsIgnoreCase(queryres.getState())
                        || 100 == queryres.getPercent())
                    {
                        log.info("QueryJobs Success.");
                        break;
                    }
                    else
                    {
                        log.debug("QueryJobs state: {} percent:{}", queryres.getState(), queryres.getPercent());
                    }

                    try { Thread.sleep(1000L); } catch (InterruptedException ignored) { }

                    //
                    if(++count>timeout)
                    {
                        queryres.setCode(TranscodeTimeoutCancelCode);
                        queryres.setMessage("水印超时取消操作: " + count);
                        log.error("SubmitJobs Failure, code:" + queryres.getCode() + " message:" + queryres.getMessage());
                        break;
                    }
                }
            }
            else
            {
                res.setCode(response.getJobResultList().get(0).getCode());
                res.setMessage(response.getJobResultList().get(0).getMessage());
                log.error("SubmitJobs Failure, code:" + res.getCode() + " message:" + res.getMessage());
            }

            return res;
        }
        catch (ClientException e)
        {
            log.error("初始配置失败 {}", e.getMessage());
            throw new ZuvException("初始配置失败", e);
        }
    }

    public MtsQueryJobRes submitQueryJob(String jobid)
    {
        QueryJobListRequest request = new QueryJobListRequest();
        request.setJobIds(jobid);

        // 发起请求并处理应答或异常
        QueryJobListResponse response;
        try
        {
            response = client.getAcsResponse(request);

            MtsQueryJobRes res = new MtsQueryJobRes();
            res.setRequestId(response.getRequestId());
//            log.debug("RequestId is:" + res.getRequestId());

            if(response.getJobList()!=null && response.getJobList().size()>0)
            {
                QueryJobListResponse.Job job = response.getJobList().get(0);
                res.setJobId(job.getJobId());
                res.setState(job.getState());  //Transcoding TranscodeSuccess
                res.setCreateTime(job.getCreationTime());
                res.setFinishTime(job.getFinishTime());
                res.setPercent(job.getPercent());

                res.setCode(job.getCode());
                res.setMessage(job.getMessage());
            }
            return res;
        }
        catch (ClientException e)
        {
            log.error("初始配置失败 {}", e.getMessage());
            throw new ZuvException("初始配置失败", e);
        }
    }

    //-----------------------------------------------------------------------------------------

}
