/*
 * Id$: zuv-cloud:z-service:cc.zuv.service.aliyun.oss.AliOssUtils:20181229114943
 *
 * AliOssUtils.java
 * Copyright (c) 2002-2020 Luther Inc.
 * http://zuv.cc
 * All rights reserved.
 */

package cc.zuv.service.aliyun.oss;

import cc.zuv.ZuvException;
import cc.zuv.service.IServiceCode;
import cc.zuv.service.storage.dfs.IDfsService;
import cc.zuv.utility.CodecUtils;
import cc.zuv.utility.MimeUtils;
import com.aliyun.oss.ClientException;
import com.aliyun.oss.OSSClient;
import com.aliyun.oss.OSSException;
import com.aliyun.oss.model.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.IOUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.io.*;
import java.util.*;

/**
 * zuv-cloud File Description
 *
 * @author          Kama Luther
 * @version         0.1
 * @since           0.1
 * @create.date     2018-12-29 11:49
 * @modify.date     2018-12-29 11:49
 */
@Slf4j
@Component
public class AliOssService implements IDfsService, IServiceCode
{

    //-----------------------------------------------------------------------------------------

    @Autowired
    private AliOssConfig config;

    //-----------------------------------------------------------------------------------------

    public String getOssUrl(String bucketkey, String filepath)
    {
        if(!config.getBuckets().containsKey(bucketkey))
        {
            throw new ZuvException("找不到对应的bucket:" + bucketkey);
        }
        AliOssConfig.Bucket bucket = config.getBuckets().get(bucketkey);
        return getOssUrl(bucket, filepath);
    }
    private String getOssUrl(AliOssConfig.Bucket bucket, String filepath)
    {
        return getOssUrl(bucket, filepath, false);
    }
    private String getOssUrl(AliOssConfig.Bucket bucket, String filepath, boolean nativecdn)
    {
        String  scheme      = config.isUsehttps()?HTTPS:HTTP;
        boolean useinter    = config.isUseinter();
        String  host        = OSS_HOST_PREFIX + bucket.getRegion();
        boolean usecdn      = bucket.isUsecdn();
        String  name        = bucket.getName();
        String  cdn         = bucket.getCdn();
        String  cdnurl      = nativecdn?getCdnNative(scheme, cdn):getCdn(scheme, cdn);
        String  endpoint    = useinter ? getInterEndpoint(scheme, name, host):getIntraEndpoint(scheme, name, host);

        String url = usecdn?cdnurl:endpoint;
        return url + "/" + filepath;
    }

    public String getBucketName(String bucketkey)
    {
        if(!config.getBuckets().containsKey(bucketkey))
        {
            throw new ZuvException("找不到对应的bucket:" + bucketkey);
        }
        AliOssConfig.Bucket bucket = config.getBuckets().get(bucketkey);
        return bucket.getName();
    }

    public String getOssEndpoint(String bucketkey)
    {
        if(!config.getBuckets().containsKey(bucketkey))
        {
            throw new ZuvException("找不到对应的bucket:" + bucketkey);
        }
        AliOssConfig.Bucket bucket = config.getBuckets().get(bucketkey);
        return getOssEndpoint(bucket);
    }
    private String getOssEndpoint(AliOssConfig.Bucket bucket)
    {
        String  scheme      = config.isUsehttps()?HTTPS:HTTP;
        boolean useinter    = config.isUseinter();
        String  host        = OSS_HOST_PREFIX + bucket.getRegion();
        return useinter?getInterEndpoint(scheme, host):getIntraEndpoint(scheme, host);
    }

    public String getMtsEndpoint(String bucketkey)
    {
        if(!config.getBuckets().containsKey(bucketkey))
        {
            throw new ZuvException("找不到对应的bucket:" + bucketkey);
        }
        AliOssConfig.Bucket bucket = config.getBuckets().get(bucketkey);
        return getMtsEndpoint(bucket);
    }
    private String getMtsEndpoint(AliOssConfig.Bucket bucket)
    {
        String  scheme      = config.isUsehttps()?HTTPS:HTTP;
        boolean useinter    = config.isUseinter();
        String  host        = MTS_HOST_PREFIX + bucket.getRegion();
        return useinter?getInterEndpoint(scheme, host):getIntraEndpoint(scheme, host);
    }

    //-----------------------------------------------------------------------------------------

    private static final String HTTP    = "http";
    private static final String HTTPS   = "https";

    //-----------------------------------------------------------------------------------------

    //cn-hangzhou、cn-shenzhen、cn-shanghai、cn-beijing

    private static final String MTS_HOST_PREFIX = "mts.";
    private static final String OSS_HOST_PREFIX = "oss-";
    private static final String OSS_HOST_INTRANET = ".aliyuncs.com";
    private static final String OSS_HOST_INTERNET = "-internal.aliyuncs.com";
    private static final String CDN_HOST = ".w.kunlungr.com";

    //-----------------------------------------------------------------------------------------

    private String getInterHost(String host)
    {
        return host + OSS_HOST_INTERNET;
    }
    private String getIntraHost(String host)
    {
        return host + OSS_HOST_INTRANET;
    }
    private String getInterHost(String bucket, String host)
    {
        return bucket + "." + getInterHost(host);
    }
    private String getIntraHost(String bucket, String host)
    {
        return bucket + "." + getIntraHost(host);
    }

    //OSS内网域名: ://oss-cn-beijing-internal.aliyuncs.com
    private String getInterEndpoint(String scheme, String host)
    {
        return scheme + "://" + getInterHost(host);
    }
    //OSS外网域名: ://oss-cn-beijing.aliyuncs.com
    private String getIntraEndpoint(String scheme, String host)
    {
        return scheme + "://" + getIntraHost(host);
    }

    //OSS内网域名: ://bucketname.oss-cn-beijing-internal.aliyuncs.com
    private String getInterEndpoint(String scheme, String bucket, String host)
    {
        return scheme + "://" + getInterHost(bucket, host);
    }
    //OSS外网域名: ://bucketname.oss-cn-beijing.aliyuncs.com
    private String getIntraEndpoint(String scheme, String bucket, String host)
    {
        return scheme + "://" + getIntraHost(bucket, host);
    }

    //CDN绑定域名: ://cdn-image.zuv.cc
    //CDN加速域名: ://cdn-image.zuv.cc.w.kunlungr.com
    //添加CNAME记录(将绑定的域名CNAME到对应的CDN的加速域名上,配置才能生效)
    private String getCdn(String scheme, String cdn)
    {
        return scheme + "://" + cdn;
    }
    public String getCdnNative(String scheme, String cdn)
    {
        return scheme + "://" + cdn + CDN_HOST;
    }

    //-----------------------------------------------------------------------------------------

    private OSSClient client;
    private AliOssConfig.Bucket bucket;

    @Override
    public void setBucketKey(String bucketkey)
    {
        if(!config.getBuckets().containsKey(bucketkey))
        {
            throw new ZuvException("找不到对应的bucket:" + bucketkey);
        }
        bucket = config.getBuckets().get(bucketkey);

        String  key         = config.getAccount().getKey();
        String  secret      = config.getAccount().getSecret();
        String  endpoint    = getOssEndpoint(bucket);
        client  = new OSSClient(endpoint, key, secret);
    }

    //-----------------------------------------------------------------------------------------

    //文件名称,包含文件后缀
    //文件目录,以'/'结尾

    private void validate()
    {
        if(bucket==null || client==null)
        {
            Map<String, AliOssConfig.Bucket> buckets = config.getBuckets();
            if(buckets !=null && buckets.size()>0)
            {
                String bucketkey = null;
                for(String key : buckets.keySet())
                {
                    bucketkey = key;
                    break;
                }

                log.info("未指定bucket,使用默认bucket: {}", bucketkey);
                setBucketKey(bucketkey);
            }
            else
            {
                throw new ZuvException("服务未初始化, 请先指定BucketKey.");
            }
        }
    }

    private void validatebucket()
    {
        if(bucket==null)
        {
            throw new ZuvException("服务未初始化, 请先指定BucketKey.");
        }
    }

    private void validatepath(String key)
    {
        if(key==null || key.startsWith("/"))
        {
            throw new ZuvException("FileKey不能为空或以'/'开头. key:" + key);
        }
    }

    private void validatefold(String key)
    {
        validatepath(key);
        if(!key.endsWith("/"))
        {
            throw new ZuvException("目录的FileKey必须以'/'结尾. key:" + key);
        }
    }

    //-----------------------------------------------------------------------------------------

    @Override
    public void initial()
    {
        log.info("initial");
    }

    @Override
    public void destroy()
    {
        log.info("destroy");
        client.shutdown();
    }

    //-----------------------------------------------------------------------------------------

    @Override
    public boolean upload(String targetpath, File sourcefile, Map<String, String> meta)
    {
        validate();
        validatepath(targetpath);

        //
        ObjectMetadata objmeta = new ObjectMetadata();
        objmeta.setContentLength(sourcefile.length());
        if(meta!=null) objmeta.setUserMetadata(meta);

        //
        String bucketname = bucket.getName();
        return upload_native(bucketname, targetpath, sourcefile, objmeta);
    }

    @Override
    public boolean upload(String targetpath, InputStream sourceis, Map<String, String> meta)
    {
        validate();
        validatepath(targetpath);

        //
        ObjectMetadata objmeta = new ObjectMetadata();
        if(meta!=null) objmeta.setUserMetadata(meta);

        //
        String bucketname = bucket.getName();
        return upload_native(bucketname, targetpath, sourceis, objmeta);
    }

    //-----------------------------------------------------------------------------------------

    @Override
    public boolean download(String sourcepath, File targetfile)
    {
        validate();
        validatepath(sourcepath);

        String bucketname = bucket.getName();
        return download_native(bucketname, sourcepath, targetfile);
    }

    @Override
    public boolean download(String sourcepath, OutputStream targetos)
    {
        validate();
        validatepath(sourcepath);

        String bucketname = bucket.getName();
        return download_native(bucketname, sourcepath, targetos);
    }

    //-----------------------------------------------------------------------------------------

    @Override
    public boolean exist(String sourcepath)
    {
        validate();
        validatepath(sourcepath);

        String bucketname = bucket.getName();
        return exist_native(bucketname, sourcepath);
    }

    //-----------------------------------------------------------------------------------------

    @Override
    public boolean movefile(String sourcepath, String targetpath)
    {
        validate();
        validatepath(sourcepath);
        validatepath(targetpath);

        String bucketname = bucket.getName();
        return move_native(bucketname, sourcepath, bucketname, targetpath);
    }

    //-----------------------------------------------------------------------------------------

    @Override
    public boolean copyfold(String sourcefold, String targetfold)
    {
        validate();
        validatefold(sourcefold);
        validatefold(targetfold);

        String bucketname = bucket.getName();
        return copy_native(bucketname, sourcefold, bucketname, targetfold);
    }

    /**
     * 拷贝文件
     * @param sourcepath 来源文件完整路径
     * @param targetpath 目标文件完整路径
     * @return 是否成功
     */
    @Override
    public boolean copyfile(String sourcepath, String targetpath)
    {
        validate();
        validatepath(sourcepath);
        validatepath(targetpath);

        String bucketname = bucket.getName();
        return copy_native(bucketname, sourcepath, bucketname, targetpath);
    }

    //-----------------------------------------------------------------------------------------

    //创建目录
    @Override
    public boolean mkfold(String sourcefold)
    {
        validate();
        validatefold(sourcefold);

        String bucketname = bucket.getName();
        return touch_native(bucketname, sourcefold);
    }

    //删除本目录下的文件及空文件夹(不带递归的删除)
    @Override
    public boolean rmfold(String sourcefold)
    {
        validate();
        validatefold(sourcefold);

        //
        List<String> keys = new ArrayList<>();
        List<Map<String, Object>> list = lsfold(sourcefold);
        for (Map<String, Object> item : list)
        {
            keys.add((String) item.get(FILE_META_FILEKEY));
        }
        keys.add(sourcefold);

        //
        String bucketname = bucket.getName();
        return delete_native(bucketname, keys);
    }

    //删除文件或目录(空目录才能删除成功,非空目录需要删除所有文件后才成功)
    @Override
    public boolean rmfile(String sourcepath)
    {
        validate();
        validatepath(sourcepath);

        String bucketname = bucket.getName();
        return delete_native(bucketname, sourcepath);
    }

    @Override
    public List<Map<String, Object>> lsfold(String sourcefold)
    {
        validate();
        validatefold(sourcefold);

        //
        String bucketname = bucket.getName();
        ListObjectsRequest request = new ListObjectsRequest(bucketname);
        request.setDelimiter("/");
        if(sourcefold!=null && !sourcefold.isEmpty())
        {
            request.setPrefix(sourcefold);
        }

        //
        List<Map<String, Object>> list = new ArrayList<>();

        ObjectListing listing = client.listObjects(request);

        log.debug("文件");
        for (OSSObjectSummary s : listing.getObjectSummaries())
        {
            String key = s.getKey();
            log.debug("\t" + key);
            if(!key.endsWith("/"))
            {
                Map<String, Object> data = new HashMap<>();
                data.put(FILE_META_FILEKEY, key);
                data.put(FILE_META_FILEKIND, FILE_KIND_FILE);
                list.add(data);
            }
        }

        log.debug("目录");
        for (String c : listing.getCommonPrefixes())
        {
            log.debug("\t" + c);
            Map<String, Object> data = new HashMap<>();
            data.put(FILE_META_FILEKEY, c);
            data.put(FILE_META_FILEKIND, FILE_KIND_FOLD);
            list.add(data);
        }

        return list;
    }

    //-----------------------------------------------------------------------------------------

    @Override
    public Map<String, Object> filemeta(String sourcepath)
    {
        validate();
        validatepath(sourcepath);

        String bucketname = bucket.getName();
        return get_meta_native(bucketname, sourcepath);
    }

    @Override
    public String filehash(String sourcepath)
    {
        validate();
        validatepath(sourcepath);

        String bucketname = bucket.getName();
        Map<String, Object> meta = get_meta_native(bucketname, sourcepath);
        return (meta.containsKey(FILE_META_FILEHASH))?(String)meta.get(FILE_META_FILEHASH):null;
    }

    @Override
    public long filedate(String sourcepath)
    {
        validate();
        validatepath(sourcepath);

        String bucketname = bucket.getName();
        Map<String, Object> meta = get_meta_native(bucketname, sourcepath);
        return (meta.containsKey(FILE_META_FILEDATE))?(Long)meta.get(FILE_META_FILEDATE):0;
    }

    @Override
    public long filesize(String sourcepath)
    {
        validate();
        validatepath(sourcepath);

        String bucketname = bucket.getName();
        Map<String, Object> meta = get_meta_native(bucketname, sourcepath);
        return (meta.containsKey(FILE_META_FILESIZE))?(Long)meta.get(FILE_META_FILESIZE):0;
    }

    //-----------------------------------------------------------------------------------------

    @Override
    public String phypath(String sourcepath)
    {
        validatebucket();

        return "oss://"+sourcepath;
    }

    @Override
    public String fileuri(String sourcepath)
    {
        validatebucket();
        validatepath(sourcepath);

        return getOssUrl(bucket,  CodecUtils.url_encode(sourcepath));
    }

    //-----------------------------------------------------------------------------------------

    public boolean upload_native(String targetbucketname, String targetpath, File sourcefile, ObjectMetadata objmeta)
    {
        try
        {
            PutObjectResult result = client.putObject(targetbucketname, targetpath, sourcefile, objmeta);
            log.info("filekey: {}, ETag: {}", targetpath, result.getETag());
            return true;
        }
        catch (OSSException | ClientException e)
        {
            log.error("上传失败 {}", e.getMessage());
            throw new ZuvException("上传失败", e);
        }
    }

    public boolean upload_native(String targetbucketname, String targetpath, InputStream sourceis, ObjectMetadata objmeta)
    {
        try
        {
            PutObjectResult result = client.putObject(targetbucketname, targetpath, sourceis, objmeta);
            log.info("filekey: {}, ETag: {}", targetpath, result.getETag());
            return true;
        }
        catch (OSSException | ClientException e)
        {
            log.error("上传失败 {}", e.getMessage());
            throw new ZuvException("上传失败", e);
        }
    }

    public boolean download_native(String sourcebucketname, String sourcepath, File targetfile)
    {
        try
        {
            ObjectMetadata objmeta = client.getObject(new GetObjectRequest(sourcebucketname, sourcepath), targetfile);
            log.info("filekey: {}, ETag: {}", sourcepath, objmeta.getETag());
            return true;
        }
        catch (OSSException | ClientException e)
        {
            log.error("下载失败 {}", e.getMessage());
            throw new ZuvException("下载失败", e);
        }
    }

    public boolean download_native(String sourcebucketname, String sourcepath, OutputStream targetos)
    {
        try
        {
            OSSObject object = client.getObject(sourcebucketname, sourcepath);
            IOUtils.copy(object.getObjectContent(), targetos);
            IOUtils.closeQuietly(targetos);
            log.info("filekey: {}, ETag: {}", sourcepath, object.getObjectMetadata().getETag());
            return true;
        }
        catch (OSSException | ClientException | IOException e)
        {
            log.error("下载失败 {}", e.getMessage());
            throw new ZuvException("下载失败", e);
        }
    }

    public boolean exist_native(String sourcebucketname, String sourcepath)
    {
        try
        {
            return client.doesObjectExist(sourcebucketname, sourcepath);
        }
        catch (OSSException | ClientException e)
        {
            log.error("查询文件失败 {}", e.getMessage());
            throw new ZuvException("查询文件失败", e);
        }
    }

    public boolean touch_native(String sourcebucketname, String sourcepath)
    {
        try
        {
            PutObjectResult result = client.putObject(sourcebucketname, sourcepath, new ByteArrayInputStream(new byte[0]));
            log.info("filekey: {}, ETag: {}", sourcepath, result.getETag());
            return true;
        }
        catch (OSSException | ClientException e)
        {
            log.error("创建文件失败 {}", e.getMessage());
            throw new ZuvException("创建文件失败", e);
        }
    }

    public boolean move_native(String sourcebucketname, String sourcepath, String targetbucketname, String targetpath)
    {
        try
        {
            CopyObjectResult result = client.copyObject(sourcebucketname, sourcepath, targetbucketname, targetpath);
            log.info("filekey: {}, ETag: {}", targetpath, result.getETag());
            client.deleteObject(sourcebucketname, sourcepath);
            return true;
        }
        catch (OSSException | ClientException e)
        {
            log.error("移动文件失败 {}", e.getMessage());
            throw new ZuvException("移动文件失败", e);
        }
    }

    public boolean copy_native(String sourcebucketname, String sourcepath, String targetbucketname, String targetpath)
    {
        try
        {
            CopyObjectResult result = client.copyObject(sourcebucketname, sourcepath, targetbucketname, targetpath);
            log.info("filekey: {}, ETag: {}", targetpath, result.getETag());
            return true;
        }
        catch (OSSException | ClientException e)
        {
            log.error("复制文件失败 {}", e.getMessage());
            throw new ZuvException("复制文件失败", e);
        }
    }

    public boolean delete_native(String sourcebucketname, List<String> keys)
    {
        try
        {
            DeleteObjectsResult result = client.deleteObjects(new DeleteObjectsRequest(sourcebucketname).withKeys(keys));
            List<String> deleteds = result.getDeletedObjects();
            return keys.size() == deleteds.size();
        }
        catch (OSSException | ClientException e)
        {
            log.error("删除文件失败 {}", e.getMessage());
            throw new ZuvException("删除文件失败", e);
        }
    }

    public boolean delete_native(String sourcebucketname, String sourcepath)
    {
        try
        {
            client.deleteObject(sourcebucketname, sourcepath);
            log.info("filepath: {}", sourcepath);
            return true;
        }
        catch (OSSException | ClientException e)
        {
            log.error("删除文件失败 {}", e.getMessage());
            throw new ZuvException("删除文件失败", e);
        }
    }

    public Map<String, Object> get_meta_native(String sourcebucketname, String sourcepath)
    {
        try
        {
            Map<String, Object> data = new HashMap<>();
            ObjectMetadata objmeta = client.getObjectMetadata(sourcebucketname, sourcepath);
            data.put(FILE_META_FILEHASH, objmeta.getContentMD5());
            data.put(FILE_META_FILESIZE, objmeta.getContentLength());
            data.put(FILE_META_FILEETAG, objmeta.getETag());
            data.put(FILE_META_FILEDATE, objmeta.getLastModified().getTime());
            data.put(FILE_META_FILEMIME, objmeta.getContentType());
            data.putAll(objmeta.getUserMetadata());
            return data;
        }
        catch (OSSException | ClientException e)
        {
            log.error("查询文件信息失败 {}", e.getMessage());
            throw new ZuvException("查询文件信息失败", e);
        }
    }

    public Map<String, String> bld_meta_native(File sourcefile)
    {
        if(!sourcefile.exists()) return null;

        String mime = MimeUtils.guessMimeByFileName(sourcefile.getAbsolutePath());
        byte filekind = sourcefile.isDirectory() ? FILE_KIND_FOLD : FILE_KIND_FILE;

        Map<String, String> data = new HashMap<>();
        data.put(FILE_META_FILENAME, sourcefile.getName());
        data.put(FILE_META_FILEHASH, CodecUtils.md5(sourcefile));
        data.put(FILE_META_FILESIZE, sourcefile.length()+"");
        data.put(FILE_META_FILEETAG, sourcefile.getName());
        data.put(FILE_META_FILEDATE, sourcefile.lastModified()+"");
        data.put(FILE_META_FILEMIME, mime);
        data.put(FILE_META_FILEKIND, filekind + "");
        return data;
    }

    //-----------------------------------------------------------------------------------------


}
