/*
 * Id$: zuv-cloud:z-service:cc.zuv.service.aliyun.search.AliSearchService:20190102113340
 *
 * AliSearchService.java
 * Copyright (c) 2002-2020 Luther Inc.
 * http://zuv.cc
 * All rights reserved.
 */

package cc.zuv.service.aliyun.search;

import cc.zuv.ZuvException;
import cc.zuv.service.search.ISearchItem;
import cc.zuv.service.search.ISearchService;
import cc.zuv.service.spider.ISpiderDocument;
import com.aliyun.opensearch.DocumentClient;
import com.aliyun.opensearch.OpenSearchClient;
import com.aliyun.opensearch.SearcherClient;
import com.aliyun.opensearch.sdk.dependencies.com.google.common.collect.Lists;
import com.aliyun.opensearch.sdk.dependencies.org.json.JSONArray;
import com.aliyun.opensearch.sdk.dependencies.org.json.JSONObject;
import com.aliyun.opensearch.sdk.generated.OpenSearch;
import com.aliyun.opensearch.sdk.generated.commons.OpenSearchClientException;
import com.aliyun.opensearch.sdk.generated.commons.OpenSearchException;
import com.aliyun.opensearch.sdk.generated.commons.OpenSearchResult;
import com.aliyun.opensearch.sdk.generated.search.*;
import com.aliyun.opensearch.sdk.generated.search.general.SearchResult;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.Map;

/**
 * zuv-cloud File Description
 *
 * @author          Kama Luther
 * @version         0.1
 * @since           0.1
 * @create.date     2019-01-02 11:33
 * @modify.date     2019-01-02 11:33
 */
@Slf4j
@Component
public class AliSearchService implements ISearchService
{

    //-----------------------------------------------------------------------------------------

    @Autowired
    private AliSearchConfig aliSearchConfig;

    private OpenSearchClient serviceClient;

    //-----------------------------------------------------------------------------------------

    private void validate()
    {
        if(aliSearchConfig==null)
        {
            throw new ZuvException("服务未初始化!");
        }

        if(serviceClient==null)
        {
            //
            String key = aliSearchConfig.getAccount().getKey();
            String secret = aliSearchConfig.getAccount().getSecret();
            String host = aliSearchConfig.getHost();
            int timeout = aliSearchConfig.getTimeout();
            int conntimeout = aliSearchConfig.getConnectTimeout();

            //
            OpenSearch openSearch = new OpenSearch(key, secret, host);
            openSearch.setTimeout(timeout);
            openSearch.setConnectTimeout(conntimeout);
            serviceClient = new OpenSearchClient(openSearch);
        }
    }

    //-----------------------------------------------------------------------------------------

    private String indexName(String indexkey)
    {
        if(aliSearchConfig==null || aliSearchConfig.getInstances()==null
            || !aliSearchConfig.getInstances().containsKey(indexkey))
        {
            throw new ZuvException("服务未初始化!");
        }

        return aliSearchConfig.getInstances().get(indexkey).getIndex();
    }

    private String tableName(String indexkey)
    {
        if(aliSearchConfig==null || aliSearchConfig.getInstances()==null
            || !aliSearchConfig.getInstances().containsKey(indexkey))
        {
            throw new ZuvException("服务未初始化!");
        }

        return aliSearchConfig.getInstances().get(indexkey).getTable();
    }

    //-----------------------------------------------------------------------------------------

    @Override
    public void initial()
    {

    }

    @Override
    public void destroy()
    {

    }

    //-----------------------------------------------------------------------------------------

    @Override
    public boolean index(String indexkey, ISpiderDocument... documents)
    {
        validate();

        //
        DocumentClient documentClient = new DocumentClient(serviceClient);

        //
        for(ISpiderDocument document : documents)
        {
            //
            Map<String, Object> data = document.build();
            if(document.deleted())
            {
                documentClient.remove(data);
            }
            else if(document.created())
            {
                documentClient.add(data);
            }
            else
            {
                documentClient.update(data);
            }
        }

        //
        String indexname = indexName(indexkey);
        String tablename = tableName(indexkey);

        //
        try
        {
            OpenSearchResult osr = documentClient.commit(indexname, tablename);
            String result = osr.getResult();
            log.info("index {} {}", result, osr.getTraceInfo());
            return "true".equalsIgnoreCase(result);
        }
        catch (OpenSearchException | OpenSearchClientException e)
        {
            log.error("index error {}", e.getMessage());
            throw new ZuvException("索引失败", e);
        }
    }

    @Override
    public List<ISearchItem> search(String indexkey, String keyword, int page, int size)
    {
        validate();

        //
        SearcherClient searcherClient = new SearcherClient(serviceClient);

        //
        String indexname = indexName(indexkey);
        Config config = new Config(Lists.newArrayList(indexname));
        config.setStart(page * size);
        config.setHits(size);
        config.setSearchFormat(SearchFormat.JSON);
//        config.setFetchFields(Lists.newArrayList("id", "title", "content"));

        //
        SearchParams searchParams = new SearchParams(config);
        searchParams.setQuery("default:" + keyword);
//        searchParams.setQuery("name:'搜索'");
//        searchParams.setFilter("id>0");

        //
        Sort sort = new Sort();
        sort.addToSortFields(new SortField("id", Order.DECREASE));
        sort.addToSortFields(new SortField("RANK", Order.INCREASE));
        searchParams.setSort(sort);

        //结果
        try
        {
            SearchResult sr = searcherClient.execute(searchParams);
            String result = sr.getResult();
            log.info("search {}", result);

            JSONObject jsonobj = new JSONObject(result);
            if(jsonobj.has("status") && jsonobj.has("result") &&
                "OK".equalsIgnoreCase(jsonobj.getString("status")))
            {
                JSONObject jsonresult = jsonobj.getJSONObject("result");

                long total = jsonresult.getLong("total");
                JSONArray items = jsonresult.getJSONArray("items");

                log.info("search total={}, items={}", total, items);

            }

            return null;
        }
        catch (OpenSearchException | OpenSearchClientException e)
        {
            log.error("search error {}", e.getMessage());
            throw new ZuvException("搜索失败", e);
        }
    }

    //{"status":"OK","request_id":"153380239219726858541484","result":{"searchtime":0.045829,"total":0,"num":0,"viewtotal":0,"compute_cost":[{"index_name":"sysdoc_dev","value":0.319}],"items":[],"facet":[]},"errors":[],"tracer":""}

    //-----------------------------------------------------------------------------------------

}
