/*
 * Id$: zuv-cloud:z-service:cc.zuv.service.mailer.impl.MailerService:20181225151549
 *
 * MailerService.java
 * Copyright (c) 2002-2020 Luther Inc.
 * http://zuv.cc
 * All rights reserved.
 */

package cc.zuv.service.mailer.impl;

import cc.zuv.ZuvException;
import cc.zuv.service.mailer.IMailerService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.*;
import org.springframework.mail.MailException;
import org.springframework.mail.SimpleMailMessage;
import org.springframework.mail.javamail.JavaMailSender;
import org.springframework.mail.javamail.MimeMessageHelper;
import org.springframework.stereotype.Component;

import javax.mail.MessagingException;
import javax.mail.internet.MimeMessage;
import java.io.File;
import java.io.InputStream;
import java.net.URL;

/**
 * File Description
 *
 * @author			Kama Luther
 * @version			0.1
 * @since           0.1
 * @create.date     2014-2-11 下午02:19:48
 * @modify.date     2014-2-11 下午02:19:48
 */
@Slf4j
@Component
public class MailerService implements IMailerService
{

    //-----------------------------------------------------------------------------------------

    @Autowired
    private JavaMailSender mailSender;

    @Value("${mail.from.addr}")
    private String from;

    //-----------------------------------------------------------------------------------------

    /**
     * 发送普通文本邮件
     */
    @Override
    public boolean sendText(String to, String subject, String content)
    {
        return sendText(from, to, subject, content);
    }

    @Override
    public boolean sendText(String[] tos, String subject, String content)
    {
        return sendText(from, tos, null, null, subject, content);
    }

    @Override
    public boolean sendText(String[] tos, String[] ccs, String[] bccs, String subject, String content)
    {
        return sendText(from, tos, ccs, bccs, subject, content);
    }

    @Override
    public boolean sendText(String from, String to, String subject, String content)
    {
        return sendText(from, new String[]{to}, null, null, subject, content);
    }

    @Override
    public boolean sendText(String from, String[] tos, String[] ccs, String[] bccs, String subject, String content)
    {
        SimpleMailMessage message = new SimpleMailMessage();
        message.setFrom(from);
        message.setTo(tos);
        if(ccs!=null && ccs.length>0) message.setCc(ccs);
        if(bccs!=null && bccs.length>0) message.setBcc(bccs);
        message.setSubject(subject);
        message.setText(content);

        try
        {
            mailSender.send(message);
            return true;
        }
        catch (MailException e)
        {
            log.error("Mail Error : " + e.getMessage());
            throw new ZuvException("mail send error " + e.getMessage());
        }
    }

    //-----------------------------------------------------------------------------------------

    /**
     * 发送富文本邮件
     */
    @Override
    public boolean sendHtml(String to, String subject, String content)
    {
        return sendHtml(from, to, subject, content);
    }

    @Override
    public boolean sendHtml(String[] tos, String subject, String content)
    {
        return sendHtml(from, tos, null, null, subject, content);
    }

    @Override
    public boolean sendHtml(String[] tos, String[] ccs, String[] bccs, String subject, String content)
    {
        return sendHtml(from, tos, ccs, bccs, subject, content);
    }

    @Override
    public boolean sendHtml(String from, String to, String subject, String content)
    {
        return sendHtml(from, new String[]{to}, null, null, subject, content);
    }

    @Override
    public boolean sendHtml(String from, String[] tos, String[] ccs, String[] bccs, String subject, String content)
    {
        MimeMessage message = mailSender.createMimeMessage();

        try
        {
            MimeMessageHelper helper = new MimeMessageHelper(message, true);
//            MimeMessageHelper helper = new MimeMessageHelper(message, false, "UTF-8");
            helper.setFrom(from);
            helper.setTo(tos);
            if(ccs!=null && ccs.length>0) helper.setCc(ccs);
            if(bccs!=null && bccs.length>0) helper.setBcc(bccs);
            helper.setSubject(subject);
            helper.setText(content, true);

            mailSender.send(message);
            return true;
        }
        catch(MailException | MessagingException e)
        {
            log.error("Mail Error : " + e.getMessage());
            throw new ZuvException("mail send error " + e.getMessage());
        }
    }

    //-------------------------------------------------------------------------------------------

    /**
     * 发送有静态资源(图片)的邮件
     */
    @Override
    public boolean sendHtmlWithInline(String to, String subject, String content, String cid, File file)
    {
        return sendHtmlWithInline(from, to, subject, content, cid, file);
    }

    @Override
    public boolean sendHtmlWithInline(String[] tos, String subject, String content, String cid, File file)
    {
        return sendHtmlWithInline(from, tos, null, null, subject, content, cid, file);
    }

    @Override
    public boolean sendHtmlWithInline(String from, String to, String subject, String content, String cid, File file)
    {
        return sendHtmlWithInline(from, new String[]{to}, null, null, subject, content, cid, file);
    }

    @Override
    public boolean sendHtmlWithInline(String from, String[] tos, String[] ccs, String[] bccs, String subject, String content, String cid, File file)
    {
        FileSystemResource resource = new FileSystemResource(file);
        return sendHtmlWithInline(from, tos, ccs, bccs, subject, content, cid, resource);
    }

    @Override
    public boolean sendHtmlWithInline(String to, String subject, String content, String cid, URL attachurl)
    {
        return sendHtmlWithInline(from, to, subject, content, cid, attachurl);
    }

    @Override
    public boolean sendHtmlWithInline(String[] tos, String subject, String content, String cid, URL attachurl)
    {
        return sendHtmlWithInline(from, tos, null, null, subject, content, cid, attachurl);
    }

    @Override
    public boolean sendHtmlWithInline(String from, String to, String subject, String content, String cid, URL attachurl)
    {
        return sendHtmlWithInline(from, new String[]{to}, null, null, subject, content, cid, attachurl);
    }

    @Override
    public boolean sendHtmlWithInline(String from, String[] tos, String[] ccs, String[] bccs, String subject, String content, String cid, URL attachurl)
    {
        UrlResource resource = new UrlResource(attachurl);
        return sendHtmlWithInline(from, tos, ccs, bccs, subject, content, cid, resource);
    }

    @Override
    public boolean sendHtmlWithInline(String to, String subject, String content, String cid, InputStream stream)
    {
        return sendHtmlWithInline(from, to, subject, content, cid, stream);
    }

    @Override
    public boolean sendHtmlWithInline(String[] tos, String subject, String content, String cid, InputStream stream)
    {
        return sendHtmlWithInline(from, tos, null, null, subject, content, cid, stream);
    }

    @Override
    public boolean sendHtmlWithInline(String from, String to, String subject, String content, String cid, InputStream stream)
    {
        return sendHtmlWithInline(from, new String[]{to}, null, null, subject, content, cid, stream);
    }

    @Override
    public boolean sendHtmlWithInline(String from, String[] tos, String[] ccs, String[] bccs, String subject, String content, String cid, InputStream stream)
    {
        InputStreamResource resource = new InputStreamResource(stream);
        return sendHtmlWithInline(from, tos, ccs, bccs, subject, content, cid, resource);
    }

    private boolean sendHtmlWithInline(String from, String[] tos, String[] ccs, String[] bccs, String subject, String content, String inlineContentId, Resource resource)
    {
        try
        {
            MimeMessage message = mailSender.createMimeMessage();
            MimeMessageHelper helper = new MimeMessageHelper(message, true, "UTF-8");
            helper.setFrom(from);
            helper.setTo(tos);
            if(ccs!=null && ccs.length>0) helper.setCc(ccs);
            if(bccs!=null && bccs.length>0) helper.setBcc(bccs);
            helper.setSubject(subject);
            helper.setText(content, true);

            //Content="<html><head></head><body><img src=\"cid:image\"/></body></html>";
            //图片：<img src='cid:image'/>
            helper.addInline(inlineContentId, resource);

            mailSender.send(message);
            return true;
        }
        catch(MailException | MessagingException e)
        {
            log.error("Mail Error : " + e.getMessage());
            throw new ZuvException("mail send error " + e.getMessage());
        }
    }

    //-----------------------------------------------------------------------------------------

    /**
     * 发送带附件的邮件
     */
    @Override
    public boolean sendHtmlWithAttach(String to, String subject, String content, File file)
    {
        return sendHtmlWithAttach(from, to, subject, content, file);
    }

    @Override
    public boolean sendHtmlWithAttach(String[] tos, String subject, String content, File file)
    {
        return sendHtmlWithAttach(from, tos, null, null, subject, content, file);
    }

    @Override
    public boolean sendHtmlWithAttach(String from, String to, String subject, String content, File file)
    {
        return sendHtmlWithAttach(from, new String[]{to}, null, null, subject, content, file);
    }

    @Override
    public boolean sendHtmlWithAttach(String from, String[] tos, String[] ccs, String[] bccs, String subject, String content, File file)
    {
        FileSystemResource resource = new FileSystemResource(file);
        return sendHtmlWithAttach(from, tos, ccs, bccs, subject, content, file.getName(), resource);
    }

    @Override
    public boolean sendHtmlWithAttach(String to, String subject, String content, String attachname, URL attachurl)
    {
        return sendHtmlWithAttach(from, to, subject, content, attachname, attachurl);
    }

    @Override
    public boolean sendHtmlWithAttach(String[] tos, String subject, String content, String attachname, URL attachurl)
    {
        return sendHtmlWithAttach(from, tos, null, null, subject, content, attachname, attachurl);
    }

    @Override
    public boolean sendHtmlWithAttach(String from, String to, String subject, String content, String attachname, URL attachurl)
    {
        return sendHtmlWithAttach(from, new String[]{to}, null, null, subject, content, attachname, attachurl);
    }

    @Override
    public boolean sendHtmlWithAttach(String from, String[] tos, String[] ccs, String[] bccs, String subject, String content, String attachname, URL attachurl)
    {
        UrlResource resource = new UrlResource(attachurl);
        return sendHtmlWithAttach(from, tos, ccs, bccs, subject, content, attachname, resource);
    }

    @Override
    public boolean sendHtmlWithAttach(String to, String subject, String content, String attachname, InputStream stream)
    {
        return sendHtmlWithAttach(from, to, subject, content, attachname, stream);
    }

    @Override
    public boolean sendHtmlWithAttach(String[] tos, String subject, String content, String attachname, InputStream stream)
    {
        return sendHtmlWithAttach(from, tos, null, null, subject, content, attachname, stream);
    }

    @Override
    public boolean sendHtmlWithAttach(String from, String to, String subject, String content, String attachname, InputStream stream)
    {
        return sendHtmlWithAttach(from, new String[]{to}, null, null, subject, content, attachname, stream);
    }

    @Override
    public boolean sendHtmlWithAttach(String from, String[] tos, String[] ccs, String[] bccs, String subject, String content, String attachname, InputStream stream)
    {
        InputStreamResource resource = new InputStreamResource(stream);
        return sendHtmlWithAttach(from, tos, ccs, bccs, subject, content, attachname, resource);
    }

    //attachname 可以任意, 不与实际文件名
    private boolean sendHtmlWithAttach(String from, String[] tos, String[] ccs, String[] bccs, String subject, String content, String attachmentFilename, InputStreamSource inputStreamSource)
    {
        //解决附件长文件名生成类似 tcmime.1138.1544.47995.bin 问题
        System.setProperty("mail.mime.splitlongparameters","false");

        MimeMessage message = mailSender.createMimeMessage();

        try
        {
            MimeMessageHelper helper = new MimeMessageHelper(message, true, "UTF-8");
            helper.setFrom(from);
            helper.setTo(tos);
            if(ccs!=null && ccs.length>0) helper.setCc(ccs);
            if(bccs!=null && bccs.length>0) helper.setBcc(bccs);
            helper.setSubject(subject);
            helper.setText(content, true);
            helper.addAttachment(attachmentFilename, inputStreamSource);

            mailSender.send(message);
            return true;
        }
        catch(MailException | MessagingException e)
        {
            log.error("Mail Error : " + e.getMessage());
            throw new ZuvException("mail send error " + e.getMessage());
        }
    }

    //-------------------------------------------------------------------------------------------

}
