/*
 * Id$: zuv-cloud:z-service:cc.zuv.service.pusher.getui.GeTuiParser:20181225151549
 *
 * GeTuiParser.java
 * Copyright (c) 2002-2020 Luther Inc.
 * http://zuv.cc
 * All rights reserved.
 */
package cc.zuv.service.pusher.getui;

import cc.zuv.collections.ArrayUtils;
import com.gexin.rp.sdk.base.IAliasResult;
import com.gexin.rp.sdk.base.IPushResult;
import com.gexin.rp.sdk.base.IQueryResult;
import com.gexin.rp.sdk.base.ITemplate;
import com.gexin.rp.sdk.base.impl.AppMessage;
import com.gexin.rp.sdk.base.impl.ListMessage;
import com.gexin.rp.sdk.base.impl.SingleMessage;
import com.gexin.rp.sdk.base.impl.Target;
import com.gexin.rp.sdk.base.payload.APNPayload;
import com.gexin.rp.sdk.base.uitls.AppConditions;
import com.gexin.rp.sdk.exceptions.RequestException;
import com.gexin.rp.sdk.http.IGtPush;
import com.gexin.rp.sdk.template.AbstractTemplate;
import com.gexin.rp.sdk.template.LinkTemplate;
import com.gexin.rp.sdk.template.NotificationTemplate;
import com.gexin.rp.sdk.template.TransmissionTemplate;
import com.gexin.rp.sdk.template.style.Style0;
import lombok.extern.slf4j.Slf4j;

import java.io.IOException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.List;

/**
 * 个推实现
 *
 * http://docs.getui.com
 *
 *
 * 一个cid对应一个alias,一个alias对应多个cid
 * 一个cid对应多个tag,一个tag对应多个cid
 *
 * @author          Kama Luther
 * @version         0.1
 * @since           0.1
 * @create.date     2018-08-30 19:05
 * @modify.date     2018-08-30 19:05
 */
@Slf4j
public class GeTuiParser
{

    //-----------------------------------------------------------------------------------------

	public GeTuiParser(String url, String appid, String appkey, String mastersecret)
	{
        this.appid = appid;
        this.appkey = appkey;

        init(url, appkey, mastersecret);
	}

    //-----------------------------------------------------------------------------------------

    private String appid;
    private String appkey;
    private IGtPush push;

    private void init(String url, String appkey, String mastersecret)
    {
        // 新建一个IGtPush实例
        try
        {
            push = new IGtPush(url, appkey, mastersecret);
            push.connect();
        }
        catch (IOException e)
        {
            log.error("init error : {}", e.getMessage());
            push = null;
        }
    }

    //-----------------------------------------------------------------------------------------

    //离线有效时间，单位为毫秒，可选
    private long offlineExipreTime = 2 * 1000 * 3600;

    public void setOfflineExipreTime(long time)
    {
        this.offlineExipreTime = time;
    }

    //-----------------------------------------------------------------------------------------

    /**
     * 发送单个终端
     * @param data 数据
     * @param cid 客户端标识
     * @return 结果
     */
    public String sendToClientId(ITemplate data, String cid)
    {
        SingleMessage message = new SingleMessage();
        message.setOffline(true);
        message.setOfflineExpireTime(offlineExipreTime);
        message.setData(data);

        Target target = new Target();
        target.setAppId(appid);
        target.setClientId(cid);

        try
        {
            IPushResult ret = push.pushMessageToSingle(message, target);
            return ret.getResponse().toString();
        }
        catch (RequestException e)
        {
            String reqid = e.getRequestId();
            IPushResult ret = push.pushMessageToSingle(message, target, reqid);
            return ret.getResponse().toString();
        }
    }

    /**
     * 发送多个终端
     * @param data 数据
     * @param cids 客户端标识
     * @return 结果
     */
    public String sendToClientIds(ITemplate data, String... cids)
    {
        ListMessage message = new ListMessage();
        message.setData(data);
        message.setOffline(true);
        message.setOfflineExpireTime(offlineExipreTime);

        List<Target> targets = new ArrayList<>();
        for(String cid : cids)
        {
            Target target = new Target();
            target.setAppId(appid);
            target.setClientId(cid);
            targets.add(target);
        }

        String taskid = push.getContentId(message);
        log.info("taskid {}", taskid);
        IPushResult ret = push.pushMessageToList(taskid, targets);
        return ret.getResponse().toString();
    }

    //-----------------------------------------------------------------------------------------

    /**
     * 发送到别名
     * @param data 数据
     * @param alias 别名
     * @return 结果
     */
    public String sendToAlias(ITemplate data, String alias)
    {
        SingleMessage message = new SingleMessage();
        message.setOffline(true);
        message.setOfflineExpireTime(offlineExipreTime);
        message.setData(data);

        Target target = new Target();
        target.setAppId(appid);
        target.setAlias(alias);

        try
        {
            IPushResult ret = push.pushMessageToSingle(message, target);
            return ret.getResponse().toString();
        }
        catch (RequestException e)
        {
            String reqid = e.getRequestId();
            IPushResult ret = push.pushMessageToSingle(message, target, reqid);
            return ret.getResponse().toString();
        }
    }

    /**
     * 发送到多个别名
     * @param data 数据
     * @param aliases 别名
     * @return 结果
     */
    public String sendToAliases(ITemplate data, String... aliases)
    {
        ListMessage message = new ListMessage();
        message.setData(data);
        message.setOffline(true);
        message.setOfflineExpireTime(offlineExipreTime);

        List<Target> targets = new ArrayList<>();
        for(String alias : aliases)
        {
            Target target = new Target();
            target.setAppId(appid);
            target.setAlias(alias);
            targets.add(target);
        }

        String taskid = push.getContentId(message);
        log.info("taskid {}", taskid);
        IPushResult ret = push.pushMessageToList(taskid, targets);
        return ret.getResponse().toString();
    }

    //-----------------------------------------------------------------------------------------

    /**
     * 发送到标签
     * @param data 数据
     * @param tags 标签
     * @return 结果
     */
    public String sendToApp(ITemplate data, String... tags)
    {
        //
        AppMessage message = new AppMessage();
        message.setData(data);
        message.setOffline(true);
        message.setOfflineExpireTime(offlineExipreTime);

        //
        List<String> appIds = new ArrayList<>();
        appIds.add(appid);
        message.setAppIdList(appIds);

        AppConditions cdt = new AppConditions();
        cdt.addCondition(AppConditions.TAG, ArrayUtils.toList(tags), AppConditions.OptType.and);
        message.setConditions(cdt);

        //
        IPushResult ret = push.pushMessageToApp(message);
        return ret.getResponse().toString();
    }

    /**
     * 定时发送到标签
     * @param pushtime 时间格式为yyyyMMddHHmm "201710261050"
     * @param groupname 任务分组名称
     * @param data 数据
     * @param tags 标签
     * @return 结果
     */
    public String sendToApp(String pushtime, String groupname, ITemplate data, String... tags)
    {
        //
        AppMessage message = new AppMessage();
        message.setData(data);
        message.setOffline(true);
        message.setOfflineExpireTime(offlineExipreTime);

        try
        {
            message.setPushTime(pushtime);
        }
        catch (ParseException e)
        {
            log.error("PushTime invalid {}", pushtime);
        }

        //
        List<String> appIds = new ArrayList<>();
        appIds.add(appid);
        message.setAppIdList(appIds);

        AppConditions cdt = new AppConditions();
        cdt.addCondition(AppConditions.TAG, ArrayUtils.toList(tags), AppConditions.OptType.and);
        message.setConditions(cdt);

        //
        IPushResult ret = push.pushMessageToApp(message, groupname);
        return ret.getResponse().toString();
    }


    /*
    List<String> phoneTypeList = new ArrayList<>(); //手机类型
    phoneTypeList.add("ANDROID");
    phoneTypeList.add("IOS");
    cdt.addCondition(AppConditions.PHONE_TYPE, phoneTypeList);

    List<String> provinceList = new ArrayList<>(); //省份
    provinceList.add("浙江");
    provinceList.add("上海");
    provinceList.add("北京");
    provinceList.add("33010000");//杭州市
    provinceList.add("51010000");//成都市
    cdt.addCondition(AppConditions.REGION, provinceList);
    */

    //-----------------------------------------------------------------------------------------

    private static int _LAUNCHED_APP = 1;    //立即启动应用
    private static int _WAITTING_APP = 2;    //等待启动应用

    /**
     * 透传消息模板
     * @param launch 是否立即启动
     * @param tcontent 透传内容
     * @return 模板
     */
    public TransmissionTemplate getTransmissionTemplate(boolean launch, String tcontent)
    {
        TransmissionTemplate template = new TransmissionTemplate();
        template.setAppId(appid);
        template.setAppkey(appkey);
        template.setTransmissionType(launch? _LAUNCHED_APP : _WAITTING_APP);
        template.setTransmissionContent(tcontent);  //透传内容
        return template;
    }

    /**
     * 链接消息模板
     * @param url 链接地址
     * @param title 通知栏标题
     * @param context 通知栏内容
     * @param logourl 通知栏图标地址
     * @return 模板
     */
    public LinkTemplate getLinkTemplate(String url, String title, String context, String logourl)
    {
        LinkTemplate template = new LinkTemplate();
        template.setAppId(appid);
        template.setAppkey(appkey);
        template.setUrl(url);

        Style0 style = new Style0();
        style.setTitle(title);
        style.setText(context);
        style.setLogoUrl(logourl);
        style.setRing(true);
        style.setVibrate(true);
        style.setClearable(true);
        template.setStyle(style);

        return template;
    }

    /**
     * 通知消息模板
     * @param title 通知栏标题
     * @param context 通知栏内容
     * @param logourl 通知栏图标地址
     * @param launch 是否立即启动
     * @param tcontent 透传内容
     * @return 模板
     */
    public NotificationTemplate getNotificationTemplate(String title, String context, String logourl, boolean launch, String tcontent)
    {
        NotificationTemplate template = new NotificationTemplate();
        template.setAppId(appid);
        template.setAppkey(appkey);
        template.setTransmissionType(launch? _LAUNCHED_APP : _WAITTING_APP);
        template.setTransmissionContent(tcontent);

        Style0 style = new Style0();
        style.setTitle(title);
        style.setText(context);
        style.setLogoUrl(logourl);
        style.setRing(true);
        style.setVibrate(true);
        style.setClearable(true);
        template.setStyle(style);

        return template;
    }

    //-----------------------------------------------------------------------------------------

    /**
     * 苹果推送
     * @param template 模板
     * @param title 推送标题
     * @param content 推送内容
     * @return 模板
     */
    public ITemplate bldAPNPayload(AbstractTemplate template, String title, String content)
    {
        APNPayload payload = new APNPayload();
        //payload.setBadge(1); 这里换setAutoBadge有没有影响 调试
        payload.setAutoBadge("+1");
        //payload.setContentAvailable(1);
        payload.setSound("default");
        //payload.setCategory("$由客户端定义");
        payload.addCustomMsg("content", content);
        payload.setAlertMsg(new APNPayload.SimpleAlertMsg(title));
        //字典模式使用下者
        //payload.setAlertMsg(getDictionaryAlertMsg());

        template.setAPNInfo(payload);
        return template;
    }

    //-----------------------------------------------------------------------------------------

    /**
     * 设置iOS手机角标
     * @param badge -n,+n,n
     * @param cids 客户端标识
     * @return 是否成功
     */
    public boolean setIOSClientBadge(String badge, String... cids)
    {
        IQueryResult result = push.setBadgeForCID(badge, appid, ArrayUtils.toList(cids));
        if(result.getResponse()!=null && result.getResponse().containsKey("result"))
        {
            String res_result = (String)result.getResponse().get("result");
            if(!"Success".equalsIgnoreCase(res_result))
            {
                log.info("error {}", result.getResponse().toString());
            }
            return "Success".equalsIgnoreCase(res_result);
        }
        return false;
    }

    //-----------------------------------------------------------------------------------------

    public String getClientAlias(String cid)
    {
        IAliasResult result = push.queryAlias(appid, cid);
        if(!result.getResult())
        {
            log.info("error {}", result.getErrorMsg());
        }
        return result.getAlias();
    }

    public boolean bindClientAlias(String cid, String alias)
    {
        IAliasResult result = push.bindAlias(appid, alias, cid);
        if(!result.getResult())
        {
            log.info("error {}", result.getErrorMsg());
        }
        return result.getResult();
    }

    public boolean unbindClientAlias(String cid, String alias)
    {
        IAliasResult result = push.unBindAlias(appid, alias, cid);
        if(!result.getResult())
        {
            log.info("error {}", result.getErrorMsg());
        }
        return result.getResult();
    }

    public List<String> getClientIdsByAlias(String alias)
    {
        IAliasResult result = push.queryClientId(appid, alias);
        if(!result.getResult())
        {
            log.info("error {}", result.getErrorMsg());
        }
        return result.getClientIdList();
    }

    public boolean unbindAliasAll(String alias)
    {
        IAliasResult result = push.unBindAliasAll(appid, alias);
        if(!result.getResult())
        {
            log.info("error {}", result.getErrorMsg());
        }
        return result.getResult();
    }

    //-----------------------------------------------------------------------------------------

    //OverLimit	每个clientId在24小时内只能设置一次
    public boolean setClientTags(String cid, String... tags)
    {

        IQueryResult result = push.setClientTag(appid, cid, ArrayUtils.toList(tags));
        if(result.getResponse()!=null && result.getResponse().containsKey("result"))
        {
            String res_result = (String)result.getResponse().get("result");
            if(!"Success".equalsIgnoreCase(res_result))
            {
                log.info("error {}", result.getResponse().toString());
            }
            return "Success".equalsIgnoreCase(res_result);
        }
        return false;
    }

    public String[] getClientTags(String cid)
    {
        IPushResult result = push.getUserTags(appid, cid);

        if(result.getResponse()!=null && result.getResponse().containsKey("result"))
        {
            String res_result = (String)result.getResponse().get("result");
            if(!"ok".equalsIgnoreCase(res_result))
            {
                log.info("error {}", result.getResponse().toString());
            }

            String res_tags = (String)result.getResponse().get("tags");
            if(res_tags!=null)
            {
                return res_tags.split(" ");
            }
        }

        return null;
    }

    //-----------------------------------------------------------------------------------------

    public boolean getClientStatusOnline(String cid)
    {
        IQueryResult result = push.getClientIdStatus(appid, cid);

        if(result.getResponse()!=null && result.getResponse().containsKey("result"))
        {
            String res_result = (String) result.getResponse().get("result");
            return "Online".equalsIgnoreCase(res_result);  //Offline
        }

        return false;
    }

    //-----------------------------------------------------------------------------------------

}
