/*
 * Id$: zuv-cloud:z-service:cc.zuv.service.pusher.huanx.HuanXinParser:20190325164948
 *
 * HuanXinParser.java
 * Copyright (c) 2002-2020 Luther Inc.
 * http://zuv.cc
 * All rights reserved.
 */
package cc.zuv.service.pusher.huanx;

import cc.zuv.ios.httpconn.IHttpConn;
import cc.zuv.ios.httpconn.IHttpRes;
import cc.zuv.ios.httpconn.httpok.OkHttpConn;
import cc.zuv.lang.StringUtils;
import cc.zuv.service.pusher.huanx.message.*;
import cc.zuv.utility.CodecUtils;
import lombok.extern.slf4j.Slf4j;

/**
 * 环信实现
 *
 * http://docs.easemob.com/
 *
 * @author          Kama Luther
 * @version         0.1
 * @since           0.1
 * @create.date     2018-08-30 19:05
 * @modify.date     2018-08-30 19:05
 */
@Slf4j
public class HuanXinParser
{

	//-------------------------------------------------------------------------------------------

	public HuanXinParser(String serverurl, String orgname, String appname,
						 String clientid, String clientsecret, boolean production)
	{

        this.srvurl = serverurl + "/" + orgname + "/" + appname;
		this.clientId = clientid;
		this.clientSecret = clientsecret;
		this.production = production;
	}

	//-------------------------------------------------------------------------------------------

    private String srvurl;
    public String clientId;
    public String clientSecret;
    public boolean production;

	private String 	accesstoken 	= null;
	private long 	expiresin 		= 0;

	//-------------------------------------------------------------------------------------------

	public void check_token()
	{
		if(StringUtils.IsEmpty(accesstoken) || expiresin<=System.currentTimeMillis())
		{
			TokenResponse tokenres = parser_token();
			if(tokenres!=null)
			{
				accesstoken 	= tokenres.getAccess_token();
				expiresin 		= tokenres.getExpires_in();
				log.info("accesstoken={} expiresin={}", accesstoken, expiresin);
			}
		}
	}

	public TokenResponse parser_token()
	{
		TokenRequest req = new TokenRequest();
		req.grant_type = "client_credentials";
		req.client_id = clientId;
		req.client_secret = clientSecret;

		String url = srvurl + "/token";

        IHttpRes response = OkHttpConn.conn(url)
            .header("Accept", "application/json")
            .header("Content-Type", "application/json;charset=utf-8")
            .mime_json("utf-8")
            .data(req)
            .timeout(60, 60, 60)
            .post();

        int status = response.status();
        log.info("status : {}", status);

        if(response.success())
        {
            return response.object(TokenResponse.class);
        }
        else if(status==401)
        {
            TokenResponse res = new TokenResponse();
            res.setExpires_in(0);
            return res;
        }
		return null;
	}

    public enum METHOD
    {
        GET,
        POST,
        PUT,
        DELETE
    }

    private <T> T getResponse(String url, METHOD method, Object request, Class<T> clazz)
    {
        //
        IHttpConn conn = OkHttpConn.conn(url)
            .header("Accept", "application/json")
            .header("Content-Type", "application/json;charset=utf-8")
            .mime_json("utf-8")
            .timeout(3000, 3000, 3000);

        if(request!=null)
        {
            conn.data(request);
        }

        if(StringUtils.NotEmpty(accesstoken))
        {
            conn.authorize_bearer(accesstoken);
        }

        //
        IHttpRes response;
        switch (method)
        {
            case POST:
                response = conn.post();
                break;
            case PUT:
                response = conn.put();
                break;
            case DELETE:
                response = conn.delete();
                break;
            default:
                response = conn.get();
                break;
        }

        //
        int status = response.status();
        log.info("status : {}", status);
        String data = response.string();
        log.info("data : {}", data);
        if(response.success())
        {
            return response.object(clazz);
        }
        return null;
    }

	public UserResponse parser_user_add(String usercode, String userpass, String usernick)
	{
		check_token();

		UserAddRequest req = new UserAddRequest();
		req.username = usercode;
		req.password = CodecUtils.sha1(userpass);
		req.nickname = usernick;

		String url = srvurl + "/users";
        return getResponse(url, METHOD.POST, req, UserResponse.class);
	}

	public UserResponse parser_user_del(String usercode)
	{
		check_token();

		String url = srvurl + "/users/" + usercode;
        return getResponse(url, METHOD.DELETE, null, UserResponse.class);
	}

	public UserResponse parser_user_chgpass(String usercode, String newpass)
	{
		check_token();

		UserChgPassRequest req = new UserChgPassRequest();
		req.newpassword = CodecUtils.sha1(newpass);

		String url = srvurl + "/users/" + usercode + "/password";
        return getResponse(url, METHOD.PUT, req, UserResponse.class);
	}

	public UserResponse parser_user_chgnick(String usercode, String usernick)
	{
		check_token();

		UserChgNickRequest req = new UserChgNickRequest();
		req.nickname = usernick;

		String url = srvurl + "/users/" + usercode;
        return getResponse(url, METHOD.PUT, req, UserResponse.class);
	}

	public UserResponse parser_user(String usercode)
	{
		check_token();

		String url = srvurl + "/users/" + usercode;
        return getResponse(url, METHOD.GET, null, UserResponse.class);
	}

	public UsersResponse parser_users(int limit, String cursor)
	{
		check_token();

		String url = srvurl + "/users?limit=" + limit;
		if(StringUtils.NotEmpty(cursor)) url = url + "&cursor=" + cursor;
        return getResponse(url, METHOD.GET, null, UsersResponse.class);
	}

	public UserResponse parser_user_status(String usercode)
	{
		check_token();

		String url = srvurl + "/users/" + usercode + "/status";
        return getResponse(url, METHOD.GET, null, UserResponse.class);
	}

	public UserResponse parser_user_disconn(String usercode)
	{
		check_token();

		String url = srvurl + "/users/" + usercode + "/disconnect";
        return getResponse(url, METHOD.GET, null, UserResponse.class);
	}

	//-------------------------------------------------------------------------------------------

	public ChatRoomResponse parser_chatroom_create(String usercode, String name, String desc, int maxu)
	{
		check_token();

		ChatRoomCreateRequest req = new ChatRoomCreateRequest();
		req.owner = usercode;
		req.name = name;
		req.description = desc;
		req.maxusers = maxu;

		String url = srvurl + "/chatrooms";
        return getResponse(url, METHOD.POST, req, ChatRoomResponse.class);
	}

	public ChatRoomResponse parser_chatroom_update(String id, String name, String desc, int maxu)
	{
		check_token();

		ChatRoomUpdateRequest req = new ChatRoomUpdateRequest();
		req.name = name;
		req.description = desc;
		req.maxusers = maxu;

		String url = srvurl + "/chatrooms/"+id;
        return getResponse(url, METHOD.PUT, req, ChatRoomResponse.class);
	}

	public ChatRoomResponse parser_chatroom_delete(String id)
	{
		check_token();

		String url = srvurl + "/chatrooms/"+id;
        return getResponse(url, METHOD.DELETE, null, ChatRoomResponse.class);
	}

	public ChatRoomResponse parser_chatroom(String id)
	{
		check_token();

		String url = srvurl + "/chatrooms/"+id;
        return getResponse(url, METHOD.GET, null, ChatRoomResponse.class);
	}

	public ChatRoomsResponse parser_chatrooms()
	{
		check_token();

		String url = srvurl + "/chatrooms";
        return getResponse(url, METHOD.GET, null, ChatRoomsResponse.class);
	}

	//-------------------------------------------------------------------------------------------

}
