/*
 * Id$: zuv-cloud:z-service:cc.zuv.service.pusher.jpush.JPushParser:20181225151549
 *
 * JPushParser.java
 * Copyright (c) 2002-2020 Luther Inc.
 * http://zuv.cc
 * All rights reserved.
 */
package cc.zuv.service.pusher.jpush;

import cc.zuv.lang.StringUtils;
import cn.jpush.api.JPushClient;
import cn.jpush.api.common.TimeUnit;
import cn.jpush.api.common.resp.APIConnectionException;
import cn.jpush.api.common.resp.APIRequestException;
import cn.jpush.api.common.resp.DefaultResult;
import cn.jpush.api.device.OnlineStatus;
import cn.jpush.api.device.TagAliasResult;
import cn.jpush.api.push.PushResult;
import cn.jpush.api.push.model.Message;
import cn.jpush.api.push.model.Options;
import cn.jpush.api.push.model.Platform;
import cn.jpush.api.push.model.PushPayload;
import cn.jpush.api.push.model.audience.Audience;
import cn.jpush.api.push.model.notification.AndroidNotification;
import cn.jpush.api.push.model.notification.IosNotification;
import cn.jpush.api.push.model.notification.Notification;
import cn.jpush.api.report.ReceivedsResult;
import cn.jpush.api.report.UsersResult;
import cn.jpush.api.schedule.ScheduleResult;
import lombok.extern.slf4j.Slf4j;

import java.util.*;

/**
 * 极光实现
 *
 * http://docs.jpush.io/server/rest_api_v3_push/
 *
 *
 * 分为
 * 1. notification: alert, title, extras 推送消息。
 * 2. message:   title, content, extras, type  应用内消息。或者称作：自定义消息，透传消息,此部分内容不会展示到通知栏上.
 * 可以二者并存
 *
 * iOS 通知 JPush 要转发给 APNs 服务器。APNs 协议定义通知长度为 2048 字节。
 * JPush 因为需要重新组包，并且考虑一点安全冗余，要求"iOS":{ } 及大括号内的总体长度不超过：2000 个字节。
 * 另外，JPush 在推送时使用 utf-8 编码，所以一个汉字占用 3 个字节长度。
 *
 * 一个用户对应一个alias,可以属于多个tag
 *
 * @author          Kama Luther
 * @version         0.1
 * @since           0.1
 * @create.date     2018-08-30 19:05
 * @modify.date     2018-08-30 19:05
 */
@Slf4j
public class JPushParser
{

    //-----------------------------------------------------------------------------------------

	public JPushParser(String appkey, String mastersecret, int maxretrytimes, boolean production)
	{
		jpushClient = new JPushClient(mastersecret, appkey, maxretrytimes);
		this.production = production;
	}

    //-----------------------------------------------------------------------------------------

	private boolean production 	= false;
	private JPushClient jpushClient;

    //-----------------------------------------------------------------------------------------

    /**
     * 发送
     * @param payload 数据
     * @return 消息ID
     */
	public long send(PushPayload payload)
	{
		try
		{
			PushResult result = jpushClient.sendPush(payload);
            if(!result.isResultOK())
            {
                log.info("result = {}", result);
            }
			return result.msg_id;
		}
		catch (APIConnectionException e)
		{
			log.error("Connection error. Should retry later. ", e);
		}
		catch (APIRequestException e)
		{
            log.info("HTTP Status ({}) : Error {} {}", e.getStatus(), e.getErrorCode(), e.getErrorMessage());
            log.debug("Msg ID: " + e.getMsgId());
		}
        return 0;
	}

    /**
     * 定时发送
     * @param payload 数据
     * @param time "2014-09-17 12:00:00"  //YYYY-MM-DD HH:MM:SS
     * @param taskname 任务名称
     * @return 任务ID
     */
    public String send(PushPayload payload, String time, String taskname)
    {
        try
        {
            ScheduleResult result = jpushClient.createSingleSchedule(taskname, time, payload);
            if(!result.isResultOK())
            {
                log.info("result = {}", result);
            }
            return result.getSchedule_id();
        }
        catch (APIConnectionException e)
        {
            log.error("Connection error. Should retry later. ", e);
        }
        catch (APIRequestException e)
        {
            log.info("HTTP Status ({}) : Error {} {}", e.getStatus(), e.getErrorCode(), e.getErrorMessage());
            log.debug("Msg ID: " + e.getMsgId());
        }
        return null;
    }

    //-----------------------------------------------------------------------------------------


    public Audience audience_all()
    {
        return Audience.all();
    }

    //alias  或关系 多个别名
    public Audience audience_alias(String... alias)
    {
        return Audience.alias(alias);
    }

    //tags  或关系 多个标签
    public Audience audience_tags_or(String... tags)
    {
        return Audience.tag(tags);
    }

    //tags  与关系 多个标签
    public Audience audience_tags_and(String... tags)
    {
        return Audience.tag_and(tags);
    }

    //regid  或关系 多个id
    public Audience audience_regid(String... regid)
    {
        return Audience.registrationId(regid);
    }

    //-----------------------------------------------------------------------------------------
    //通知
    //-----------------------------------------------------------------------------------------

	public PushPayload build_notification(Audience audience, String alert)
	{
		Notification notification = Notification.newBuilder().setAlert(alert).build();
		Options options = Options.newBuilder().setApnsProduction(production).build();//是否生产环境

		return PushPayload.newBuilder()
                .setPlatform(Platform.all())
                .setAudience(audience)
                .setNotification(notification)
                .setOptions(options)
                .build();
	}

    //-----------------------------------------------------------------------------------------
    // 消息
    //-----------------------------------------------------------------------------------------

    public PushPayload build_message(Audience audience, String content, String title, Map<String, String> extras)
    {
        Message message = Message.newBuilder().setMsgContent(content).setTitle(title).addExtras(extras).build();
        return PushPayload.newBuilder()
            .setPlatform(Platform.all())
            .setAudience(audience)
            .setMessage(message)
            .build();
    }


    //-----------------------------------------------------------------------------------------
    //通知 限定终端
    //-----------------------------------------------------------------------------------------

	public PushPayload build_android_notification(Audience audience, String alert, String title, Map<String, String> extras)
	{
		AndroidNotification builder = AndroidNotification.newBuilder().setAlert(alert).setTitle(title).addExtras(extras).build();
		Notification notification = Notification.newBuilder().addPlatformNotification(builder).build();

		return PushPayload.newBuilder()
                .setPlatform(Platform.android())
                .setAudience(audience)
                .setNotification(notification)
                .build();
	}

	public PushPayload build_ios_notification(Audience audience, String alert, Map<String, String> extras)
	{
		IosNotification builder = IosNotification.newBuilder().setAlert(alert)
            .incrBadge(1).addExtras(extras).build();  //.setBadge(5) .setBadge(0)
		Notification notification = Notification.newBuilder().addPlatformNotification(builder).build();
		Options options = Options.newBuilder().setApnsProduction(production).build(); //是否生产环境

		return PushPayload.newBuilder()
                .setPlatform(Platform.ios())
                .setAudience(audience)
                .setNotification(notification)
                .setOptions(options)
                .build();
	}

	public PushPayload build_android_ios_notification(Audience audience, String alert, String title, Map<String, String> extras)
	{
		AndroidNotification builder_android = AndroidNotification.newBuilder().setTitle(title).addExtras(extras).build();
		IosNotification builder_ios = IosNotification.newBuilder()
            .incrBadge(1).addExtras(extras).build();  //.setBadge(5) .setBadge(0)
		Notification notification = Notification.newBuilder()
				.setAlert(alert).addPlatformNotification(builder_android).addPlatformNotification(builder_ios)
				.build();
		Options options = Options.newBuilder().setApnsProduction(production).build(); //是否生产环境

		return PushPayload.newBuilder()
                .setPlatform(Platform.android_ios())
                .setAudience(audience)
                .setNotification(notification)
                .setOptions(options)
                .build();
	}

    //-----------------------------------------------------------------------------------------
    //通知&通知 限定终端
    //-----------------------------------------------------------------------------------------

    public PushPayload build_android(Audience audience, String alert, String title, String content, Map<String, String> extras)
    {
        AndroidNotification builder = AndroidNotification.newBuilder().setTitle(title).addExtras(extras).build();
        Notification notification = Notification.newBuilder().setAlert(alert).addPlatformNotification(builder).build();
        Message message = Message.newBuilder().setMsgContent(content).setTitle(title).addExtras(extras).build();

        return PushPayload.newBuilder()
            .setPlatform(Platform.android())
            .setAudience(audience)
            .setNotification(notification)
            .setMessage(message)
            .build();
    }

	public PushPayload build_ios(Audience audience, String alert, String title, String content, Map<String, String> extras)
	{
		IosNotification builder = IosNotification.newBuilder().setAlert(alert)
            .incrBadge(1).addExtras(extras).build();  //.setBadge(5) .setBadge(0)
		Notification notification = Notification.newBuilder().addPlatformNotification(builder).build();
		Message message = Message.newBuilder().setMsgContent(content).setTitle(title).addExtras(extras).build();
		Options options = Options.newBuilder().setApnsProduction(production).build(); //是否生产环境

		return PushPayload.newBuilder()
                .setPlatform(Platform.ios())
                .setAudience(audience)
                .setNotification(notification)
                .setMessage(message)
                .setOptions(options)
                .build();
	}

	public PushPayload build_android_ios(Audience audience, String alert, String title, String content, Map<String, String> extras, boolean silent, boolean apns)
	{
		String sound = silent?"":"default";
		IosNotification builder_ios = IosNotification.newBuilder()
            .incrBadge(1).addExtras(extras).setSound(sound).build();  //.setBadge(5) .setBadge(0)
		AndroidNotification builder_android = AndroidNotification.newBuilder().setTitle(title).addExtras(extras).build();
		Notification notification = Notification.newBuilder()
				.setAlert(alert)
				.addPlatformNotification(builder_android)
				.addPlatformNotification(builder_ios)
				.build();

		Message message = Message.newBuilder().setMsgContent(content).setTitle(title).addExtras(extras).build();
		Options options = apns? Options.newBuilder().setApnsProduction(production).build(): Options.newBuilder().build(); //是否生产环境

		return PushPayload.newBuilder()
                .setPlatform(Platform.android_ios())
                .setAudience(audience)
                .setNotification(notification)
                .setMessage(message)
                .setOptions(options)
                .build();
	}

    //-----------------------------------------------------------------------------------------

    public String get_alias(String regid)
    {
        try
        {
            TagAliasResult result = jpushClient.getDeviceTagAlias(regid);
            return result.alias;
        }
        catch (APIConnectionException e)
        {
            log.error("Connection error. Should retry later. ", e);
        }
        catch (APIRequestException e)
        {
            log.info("HTTP Status ({}) : Error {} {}", e.getStatus(), e.getErrorCode(), e.getErrorMessage());
            log.debug("Msg ID: " + e.getMsgId());
        }
        return null;
    }

    public boolean set_alias(String regid, String alias)
    {
        try
        {
            DefaultResult clear = jpushClient.updateDeviceTagAlias(regid, true, false);
            log.debug("clear={}", clear.isResultOK());

            DefaultResult setter = jpushClient.updateDeviceTagAlias(regid, alias, null, null);
            log.debug("setter={}", setter.isResultOK());

            return clear.isResultOK() && setter.isResultOK();
        }
        catch (APIConnectionException e)
        {
            log.error("Connection error. Should retry later. ", e);
        }
        catch (APIRequestException e)
        {
            log.info("HTTP Status ({}) : Error {} {}", e.getStatus(), e.getErrorCode(), e.getErrorMessage());
            log.debug("Msg ID: " + e.getMsgId());
        }
        return false;
    }

    public List<String> get_tags(String regid)
    {
        try
        {
            TagAliasResult result = jpushClient.getDeviceTagAlias(regid);
            return result.tags;
        }
        catch (APIConnectionException e)
        {
            log.error("Connection error. Should retry later. ", e);
        }
        catch (APIRequestException e)
        {
            log.info("HTTP Status ({}) : Error {} {}", e.getStatus(), e.getErrorCode(), e.getErrorMessage());
            log.debug("Msg ID: " + e.getMsgId());
        }
        return null;
    }

    //有效的 tag 组成：字母（区分大小写）、数字、下划线、汉字、特殊字符@!#$&*+=.|￥。
    public boolean set_tags(String regid, String... tags)
    {
        Set<String> taginsert = new TreeSet<>();
        Collections.addAll(taginsert, tags);

        try
        {
            DefaultResult clear = jpushClient.updateDeviceTagAlias(regid, false, true);
            log.debug("clear={}", clear.isResultOK());

            DefaultResult setter = jpushClient.updateDeviceTagAlias(regid, null, taginsert, null);
            log.debug("setter={}", setter.isResultOK());

            return clear.isResultOK() && setter.isResultOK();
        }
        catch (APIConnectionException e)
        {
            log.error("Connection error. Should retry later. ", e);
        }
        catch (APIRequestException e)
        {
            log.info("HTTP Status ({}) : Error {} {}", e.getStatus(), e.getErrorCode(), e.getErrorMessage());
            log.debug("Msg ID: " + e.getMsgId());
        }
        return false;
    }

	public boolean add_tag(String regid, String tag)
	{
		try
		{
            //查询当前的
			TagAliasResult getter = jpushClient.getDeviceTagAlias(regid);
            log.debug("getter={}", getter.isResultOK());

            //加入新加的
            List<String> tags = getter.tags;
			Set<String> taginsert = new TreeSet<>();
			for(String t : tags)
			{
				taginsert.add(t);
			}
			taginsert.add(tag);

			DefaultResult clear = jpushClient.updateDeviceTagAlias(regid, false, true);
			log.debug("clear={}", clear.isResultOK());

			DefaultResult setter = jpushClient.updateDeviceTagAlias(regid, null, taginsert, null);
			log.debug("setter={}", setter.isResultOK());

            return getter.isResultOK() && clear.isResultOK() && setter.isResultOK();
		}
		catch (APIConnectionException e)
		{
			log.error("Connection error. Should retry later. ", e);
		}
		catch (APIRequestException e)
		{
            log.info("HTTP Status ({}) : Error {} {}", e.getStatus(), e.getErrorCode(), e.getErrorMessage());
            log.debug("Msg ID: " + e.getMsgId());
		}
        return false;
	}

	public boolean del_tag(String regid, String tag)
	{
		try
		{
			Set<String> tagremove = new TreeSet<>();
			tagremove.add(tag);

			DefaultResult clear = jpushClient.updateDeviceTagAlias(regid, false, true);
			log.debug("clear={}", clear.isResultOK());

			DefaultResult setter = jpushClient.updateDeviceTagAlias(regid, null, null, tagremove);
			log.debug("setter={}", setter.isResultOK());

            return clear.isResultOK() && setter.isResultOK();
		}
		catch (APIConnectionException e)
		{
			log.error("Connection error. Should retry later. ", e);
		}
		catch (APIRequestException e)
		{
            log.info("HTTP Status ({}) : Error {} {}", e.getStatus(), e.getErrorCode(), e.getErrorMessage());
            log.debug("Msg ID: " + e.getMsgId());
		}
        return false;
	}

	public boolean clear_alias_tags(String regid)
	{
		try
		{
			DefaultResult clear = jpushClient.updateDeviceTagAlias(regid, true, true);
            return clear.isResultOK();
		}
		catch (APIConnectionException e)
		{
			log.error("Connection error. Should retry later. ", e);
		}
		catch (APIRequestException e)
		{
            log.info("HTTP Status ({}) : Error {} {}", e.getStatus(), e.getErrorCode(), e.getErrorMessage());
            log.debug("Msg ID: " + e.getMsgId());
		}
        return false;
    }

    //-----------------------------------------------------------------------------------------

    // 需要加白名单
    public boolean get_user_onlinestatus(String regid)
    {
        try
        {
            Map<String, OnlineStatus> result =  jpushClient.getUserOnlineStatus(regid);
            log.debug("status={}", result.get(regid).toString());
            return result.get(regid)!=null && result.get(regid).getOnline();
        }
        catch (APIConnectionException e)
        {
            log.error("Connection error. Should retry later. ", e);
        }
        catch (APIRequestException e)
        {
            log.info("HTTP Status ({}) : Error {} {}", e.getStatus(), e.getErrorCode(), e.getErrorMessage());
            log.debug("Msg ID: " + e.getMsgId());
        }
        return false;
    }

    //-----------------------------------------------------------------------------------------

    //[{"android_received":2,"ios_apns_received":null,"ios_apns_sent":null,"ios_msg_received":null,"msg_id":"3672628239","wp_mpns_sent":null}]
	public String get_message_report(String msgid)
	{
        //android_received
        //ios_apns_received
        //ios_msg_received
        //ios_apns_sent
        //wp_mpns_sent

		try
		{
			ReceivedsResult recv = jpushClient.getReportReceiveds(msgid);
            int code = recv.getResponseCode();
            if(code != 200)
            {
                log.error("recv : {}", code);
            }
            return recv.getOriginalContent();
		}
		catch (APIConnectionException e)
		{
			log.error("Connection error. Should retry later. ", e);
		}
		catch (APIRequestException e)
		{
            log.info("HTTP Status ({}) : Error {} {}", e.getStatus(), e.getErrorCode(), e.getErrorMessage());
            log.debug("Msg ID: " + e.getMsgId());
		}
        return null;
	}

	// "2014-06-10",3
    // 需要加白名单
	public void get_user_report(String start, int duration)
	{
		try
		{
			UsersResult result = jpushClient.getReportUsers(TimeUnit.DAY, start, duration);
			log.info("result={}",result);
		}
		catch (APIConnectionException e)
		{
			log.error("Connection error. Should retry later. ", e);
		}
		catch (APIRequestException e)
		{
            log.info("HTTP Status ({}) : Error {} {}", e.getStatus(), e.getErrorCode(), e.getErrorMessage());
            log.debug("Msg ID: " + e.getMsgId());
		}
	}

    //-----------------------------------------------------------------------------------------

}
