/*
 * Id$: zuv-cloud:z-service:cc.zuv.service.smser.alisms.AliSmserService:20190222164501
 *
 * AliSmserService.java
 * Copyright (c) 2002-2020 Luther Inc.
 * http://zuv.cc
 * All rights reserved.
 */

package cc.zuv.service.smser.alisms;

import cc.zuv.ZuvException;
import cc.zuv.collections.ArrayUtils;
import cc.zuv.document.support.json.GsonParser;
import cc.zuv.lang.StringUtils;
import cc.zuv.service.smser.ISmserService;
import cc.zuv.service.smser.condition.AlismsCondition;
import cc.zuv.utility.DateUtils;
import com.aliyuncs.DefaultAcsClient;
import com.aliyuncs.IAcsClient;
import com.aliyuncs.dysmsapi.model.v20170525.*;
import com.aliyuncs.exceptions.ClientException;
import com.aliyuncs.http.MethodType;
import com.aliyuncs.profile.DefaultProfile;
import com.aliyuncs.profile.IClientProfile;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Conditional;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import java.util.Date;
import java.util.List;
import java.util.Map;


/**
 * zuv-cloud File Description
 *
 * @author          Kama Luther
 * @version         0.1
 * @since           0.1
 * @create.date     2019-02-22 16:23
 * @modify.date     2019-02-22 16:23
 */
@Slf4j
@Service
@Conditional(AlismsCondition.class)
public class AliSmserService implements ISmserService
{

    //-----------------------------------------------------------------------------------------

    public static final String DATE_PATTERN = "yyyy-MM-dd HH:mm:ss";

    //-----------------------------------------------------------------------------------------

    @Value("${zuvboot.sms.configs.alisms.key}")
    private String key;
    @Value("${zuvboot.sms.configs.alisms.secret}")
    private String secret;
    @Value("${zuvboot.sms.configs.alisms.signname}")
    private String signname;

    //-----------------------------------------------------------------------------------------

    private IAcsClient acsClient;

    @PostConstruct
    private void initialize()
    {
        // 初始化ascClient需要的几个参数
        final String product = "Dysmsapi";// 短信API产品名称（短信产品名固定，无需修改）
        final String domain = "dysmsapi.aliyuncs.com";// 短信API产品域名（接口地址固定，无需修改）

        // 初始化ascClient,暂时不支持多region（请勿修改）
        IClientProfile profile = DefaultProfile.getProfile("cn-hangzhou", key, secret);
        try
        {
            DefaultProfile.addEndpoint("cn-hangzhou", "cn-hangzhou", product, domain);
        }
        catch (ClientException e)
        {
            log.error("initialize failure {}", e.getMessage());
            throw new ZuvException("initialize failure", e);
        }
        acsClient = new DefaultAcsClient(profile);
    }

    //-----------------------------------------------------------------------------------------

    /**
     * 发送单条短信
     * @param mobiles 手机号码
     * @param template 模板编号
     * @param param 模板变量,如{name:'张三','code':'222'}
     * @return 消息ID
     */
    @Override
    public String send(String[] mobiles, String template, Map<String, String> param)
    {
        //
        SendSmsRequest request = new SendSmsRequest();
        request.setConnectTimeout(10000);
        request.setReadTimeout(10000);
        request.setMethod(MethodType.POST);
        // 必填:待发送手机号。支持以逗号分隔的形式进行批量调用，批量上限为1000个手机号码,批量调用相对于单条调用及时性稍有延迟,验证码类型的短信推荐使用单条调用的方式
        request.setPhoneNumbers(ArrayUtils.concat(mobiles, ","));
        // 必填:短信签名-可在短信控制台中找到,不包含【】
        request.setSignName(signname);
        // 必填:短信模板-可在短信控制台中找到
        request.setTemplateCode(template);
        // 可选:模板中的变量替换JSON串,如模板内容为"亲爱的${name},您的验证码为${code}"时,此处的值为"{name:'张三','code':'222'}"
        request.setTemplateParam(GsonParser.json(param));
        // 可选-上行短信扩展码(扩展码字段控制在7位或以下，无特殊需求用户请忽略此字段)
        // request.setSmsUpExtendCode("90997");
        // 可选:outId为提供给业务方扩展字段,最终在短信回执消息中将此值带回给调用者
        // request.setOutId("yourOutId");

        //
        try
        {
            SendSmsResponse response = acsClient.getAcsResponse(request);
            String code = response.getCode();
            String message = response.getMessage();
            String bazid = response.getBizId();

            if(!"OK".equalsIgnoreCase(code))
            {
                log.error("send error {} {}", code, message);
                throw new ZuvException("send error" + message, code, null);
            }
            return bazid;
        }
        catch (ClientException e)
        {
            log.error("request error {}" + e.getMessage(), e);
            throw new ZuvException("request error" + e.getMessage());
        }
    }

    /**
     * 发送批量短信
     * @param mobiles 手机号码
     * @param template 短信模板编码
     * @param tplparams 短信模板参数
     * @return 消息ID
     */
    @Override
    public String send(String[] mobiles, String template, List<Map<String, String>> tplparams)
    {
        if(mobiles==null || mobiles.length==0) throw new ZuvException("参数错误");

        //
        String[] signnames = new String[mobiles.length];
        for(int i=0;i<mobiles.length;i++)
        {
            signnames[i] = signname;
        }

        //组装请求对象
        SendBatchSmsRequest request = new SendBatchSmsRequest();
        //使用post提交
        request.setMethod(MethodType.POST);
        //必填:待发送手机号。支持JSON格式的批量调用，批量上限为100个手机号码,批量调用相对于单条调用及时性稍有延迟,验证码类型的短信推荐使用单条调用的方式
        request.setPhoneNumberJson(GsonParser.json(mobiles)); //"[\"1500000000\",\"1500000001\"]"
        //必填:短信签名-支持不同的号码发送不同的短信签名
        request.setSignNameJson(GsonParser.json(signnames)); //"[\"云通信\",\"云通信\"]"
        //必填:短信模板-可在短信控制台中找到
        request.setTemplateCode(template);
        //必填:模板中的变量替换JSON串,如模板内容为"亲爱的${name},您的验证码为${code}"时,此处的值为
        //友情提示:如果JSON中需要带换行符,请参照标准的JSON协议对换行符的要求,比如短信内容中包含\r\n的情况在JSON中需要表示成\\r\\n,否则会导致JSON在服务端解析失败
        request.setTemplateParamJson(GsonParser.json(tplparams)); //"[{\"name\":\"Tom\", \"code\":\"123\"},{\"name\":\"Jack\", \"code\":\"456\"}]"
        //可选-上行短信扩展码(扩展码字段控制在7位或以下，无特殊需求用户请忽略此字段)
        //request.setSmsUpExtendCodeJson("[\"90997\",\"90998\"]");

        //
        try
        {
            SendBatchSmsResponse response = acsClient.getAcsResponse(request);
            String code = response.getCode();
            String message = response.getMessage();
            String bazid = response.getBizId();

            if(!"OK".equalsIgnoreCase(code))
            {
                log.error("send error {} {}", code, message);
                throw new ZuvException("send error" + message, code, null);
            }
            return bazid;
        }
        catch (ClientException e)
        {
            log.error("request error {}" + e.getMessage(), e);
            throw new ZuvException("request error" + e.getMessage());
        }
    }

    /**
     * 查询产品余额
     * @param productid 产品ID
     * @return 余额
     */
    @Override
    public long balance(String productid)
    {
        throw new ZuvException("不支持操作");
    }

    /**
     * 查询短信发送状态
     * @param bizid 消息ID
     * @param mobile 手机号码
     * @param time 查询日期(年月日)
     * @return 状态信息(状态值:时间)
     */
    @Override
    public String report(String bizid, String mobile, Date time)
    {
        if(StringUtils.IsEmpty(bizid)) throw new ZuvException("参数错误");

        String senddate = DateUtils.format(time, "yyyyMMdd");

        QuerySendDetailsRequest request = new QuerySendDetailsRequest();
        //必填-号码
        request.setPhoneNumber(mobile);
        //可选-调用发送短信接口时返回的BizId
        request.setBizId(bizid);
        //必填-短信发送的日期 支持30天内记录查询（可查其中一天的发送数据），格式yyyyMMdd
        request.setSendDate(senddate);
        //必填-页大小
        request.setPageSize(20L);
        //必填-当前页码从1开始计数
        request.setCurrentPage(1L);

        //
        try
        {
            QuerySendDetailsResponse response = acsClient.getAcsResponse(request);
            String code = response.getCode();
            String message = response.getMessage();

            if(!"OK".equalsIgnoreCase(code))
            {
                log.error("send error {} {}", code, message);
                throw new ZuvException("send error" + message, code, null);
            }

            List<QuerySendDetailsResponse.SmsSendDetailDTO> details = response.getSmsSendDetailDTOs();
            if(details==null || details.size()==0) return null;

            QuerySendDetailsResponse.SmsSendDetailDTO detail = details.get(0); //取第一条
            long sendstatus = detail.getSendStatus();  //发送状态 1：等待回执，2：发送失败，3：发送成功
            String errcode = detail.getErrCode(); //运营商短信错误码 DELIVERED
            String receviedate = detail.getReceiveDate(); //2017-05-25 00:00:00
            return sendstatus+"-"+errcode+":"+receviedate;
        }
        catch (ClientException e)
        {
            log.error("request error {}" + e.getMessage(), e);
            throw new ZuvException("request error" + e.getMessage());
        }
    }

    /**
     * 查询短信上行
     * @param mobile 手机号码
     * @param begintime 开始时间
     * @param endtime 结束时间
     * @return 上行短信列表(时间:上行号码:消息)
     */
    @Override
    public List<String> mo(String mobile, Date begintime, Date endtime)
    {
        throw new ZuvException("不支持操作");
    }

    //-----------------------------------------------------------------------------------------

    @Override
    public boolean receive_report(String msgid, String status, String mobile, Date stamp)
    {
        log.info("receive_report {} {} {} {}", msgid, status, mobile, stamp);
        return true;
    }

    @Override
    public boolean receive_mo(String message, String dstcode, String mobile, Date stamp)
    {
        log.info("receive_mo {} {} {} {}", message, dstcode, mobile, stamp);
        return true;
    }

    //-----------------------------------------------------------------------------------------

}
