/*
 * Id$: zuv-cloud:z-service:cc.zuv.service.smser.langyu.LySmserService:20190222164501
 *
 * LySmserService.java
 * Copyright (c) 2002-2020 Luther Inc.
 * http://zuv.cc
 * All rights reserved.
 */

package cc.zuv.service.smser.langyu;

import cc.zuv.ZuvException;
import cc.zuv.collections.ArrayUtils;
import cc.zuv.document.support.json.GsonParser;
import cc.zuv.ios.httpconn.IHttpRes;
import cc.zuv.ios.httpconn.httpok.OkHttpConn;
import cc.zuv.lang.StringUtils;
import cc.zuv.service.smser.ISmserService;
import cc.zuv.service.smser.condition.LangyuCondition;
import cc.zuv.utility.DateUtils;
import lombok.Getter;
import lombok.Setter;
import lombok.ToString;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Conditional;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import java.util.*;

/**
 * zuv-cloud File Description
 *
 * @author          Kama Luther
 * @version         0.1
 * @since           0.1
 * @create.date     2019-02-22 16:08
 * @modify.date     2019-02-22 16:08
 */
@Slf4j
@Service
@Conditional(LangyuCondition.class)
public class LySmserService implements ISmserService
{

    //-----------------------------------------------------------------------------------------

    public static final String DATE_PATTERN = "yyMMddHHmm";

    //-----------------------------------------------------------------------------------------

    @Value("${zuvboot.sms.configs.langyu.submitbat-url}")
    private String submitbatUrl;
    @Value("${zuvboot.sms.configs.langyu.submitvar-url}")
    private String submitvarUrl;
    @Value("${zuvboot.sms.configs.langyu.getbalance-url}")
    private String getbalanceUrl;
    @Value("${zuvboot.sms.configs.langyu.getreport-url}")
    private String getreportUrl;
    @Value("${zuvboot.sms.configs.langyu.account}")
    private String account;
    @Value("${zuvboot.sms.configs.langyu.password}")
    private String password;
    @Value("${zuvboot.sms.configs.langyu.product-id}")
    private String productId;
    @Value("${zuvboot.sms.configs.langyu.extno}")
    private String extno;

    //-----------------------------------------------------------------------------------------

    @PostConstruct
    private void initialize()
    {

    }

    //-----------------------------------------------------------------------------------------

    /**
     * 发送单条短信
     * @param mobiles 手机号码
     * @param template 短信内容, 含签名, 如 "【某某公司】,你的验证码是${code}"
     * @param param 替换参数, 不为空的话, 可以替换短信内容的变量, 如 {'code': '1234'}
     * @return 消息ID
     */
    @Override
    public String send(String[] mobiles, String template, Map<String, String> param)
    {
        //
        if(param!=null && param.size()>0)
        {
            for(String key : param.keySet())
            {
                template = template.replaceAll("\\$\\{"+key+"\\}", param.get(key));
            }
        }

        //
        Map<String, Object> data = new HashMap<>();
        data.put("account", account);
        data.put("pswd", password);
        data.put("needstatus", true);
        data.put("product", productId);
        data.put("extno", extno);
        data.put("mobile", ArrayUtils.concat(mobiles, ","));
        data.put("msg", template);
        data.put("resptype", "json");

        //
        IHttpRes response = OkHttpConn.conn(submitbatUrl).data(data).post();
        if(!response.success())
        {
            log.error("request error {} {}", response.status(), response.message());
            throw new ZuvException("request error" + response.message());
        }

        //
        String resstr = response.string();
        log.debug("resstr {}", resstr);
        Result result = GsonParser.fromJson(resstr, Result.class);
        log.debug("result {}", result);
        if(!result.success())
        {
            log.error("send error {}: {} {}", result.getResult(), result.getSuc(), result.getFail());
            throw new ZuvException("send error", "" + result.getResult(), null);
        }
        return result.getMsgid();
    }

    /**
     * 发送批量短信
     * @param mobiles 手机号码,多个以","号分割
     * @param template 短信模板内容, 含签名及变量,变量用"{$var}"来替代,如 "【公司】,{$var},你好,你的成绩是{$var}"
     * @param tplparams 短信模板变量,如[{'var1':'张三', 'var2':'60'},{'var1':'李四', 'var2':'70'}]
     * @return 消息ID
     */
    @Override
    public String send(String[] mobiles, String template, List<Map<String, String>> tplparams)
    {
        //
        if(mobiles==null || tplparams==null
            || mobiles.length!=tplparams.size()
            || tplparams.size()==0)
        {
            throw new ZuvException("参数错误");
        }

        //
        StringBuilder params = new StringBuilder();
        int line = mobiles.length;
        for (int i=0;i<line;i++)
        {
            params.append(mobiles[i]);
            Map<String, String> lineparams = tplparams.get(i);
            for(String v : lineparams.values())
            {
                params.append(",").append(v);
            }
            if(i<line-1) params.append(";");
        }

        //
        Map<String, Object> data = new HashMap<>();
        data.put("account", account);
        data.put("pswd", password);
        data.put("needstatus", true);
        data.put("product", productId);
        data.put("extno", extno);
        data.put("msg", template);
        data.put("params", params.toString());  //13800210000,李先生,2013-01-01;13500210000,王先生,2013-01-15
        data.put("resptype", "json");

        //
        IHttpRes response = OkHttpConn.conn(submitvarUrl).data(data).post();
        if(!response.success())
        {
            log.error("request error {} {}", response.status(), response.message());
            throw new ZuvException("request error" + response.message());
        }

        String resstr = response.string();
        log.debug("resstr {}", resstr);
        Result result = GsonParser.fromJson(resstr, Result.class);
        log.debug("result {}", result);
        if(!result.success())
        {
            log.error("send error {}: {} {}", result.getResult(), result.getSuc(), result.getFail());
            throw new ZuvException("send error", "" + result.getResult(), null);
        }
        return result.getMsgid();
    }

    /**
     * 查询产品余额
     * @param productid 产品ID
     * @return 余额
     */
    @Override
    public long balance(String productid)
    {
        if(StringUtils.IsEmpty(productid)) throw new ZuvException("参数错误");

        Map<String, Object> data = new HashMap<>();
        data.put("account", account);
        data.put("pswd", password);
        data.put("resptype", "json");

        IHttpRes response = OkHttpConn.conn(getbalanceUrl).data(data).post();
        if(!response.success())
        {
            log.error("request error {} {}", response.status(), response.message());
            throw new ZuvException("request error" + response.message());
        }

        String resstr = response.string();
        log.debug("resstr {}", resstr);
        Balance result = GsonParser.fromJson(resstr, Balance.class);
        log.debug("result {}", result);

        if(!result.success() || result.getProducts()==null) return 0;

        for(Product product : result.getProducts())
        {
            if(productid.equalsIgnoreCase(product.getProduct()))
            {
                return product.getNum();
            }
        }

        return 0;
    }

    /**
     * 查询短信发送状态
     * @param msgid 消息ID
     * @param mobile 手机号码
     * @param time 查询日期(年月日)
     * @return 状态信息(状态值:时间)
     */
    @Override
    public String report(String msgid, String mobile, Date time)
    {
        if(StringUtils.IsEmpty(msgid)) throw new ZuvException("参数错误");
        if(time==null) throw new ZuvException("参数错误");

        Map<String, Object> data = new HashMap<>();
        data.put("account", account);
        data.put("pswd", password);
        data.put("resptype", "json");

        IHttpRes response = OkHttpConn.conn(getreportUrl).data(data).post();
        if(!response.success())
        {
            log.error("request error {} {}", response.status(), response.message());
            throw new ZuvException("request error" + response.message());
        }

        String resstr = response.string();
        log.debug("resstr {}", resstr);
        Reports result = GsonParser.fromJson(resstr, Reports.class);
        log.debug("result {}", result);

        if(!result.success() || result.getReport()==null) return null;

        for(Report report : result.getReport())
        {
            //过滤时间
            String reporttime = report.getReportTime();
            if(StringUtils.IsEmpty(reporttime)) continue;

            Date reportdate = DateUtils.format(reporttime, DATE_PATTERN);
            String filterdate = DateUtils.format(time, DateUtils.PATTERN_DATE);
            if(!DateUtils.format(reportdate, DateUtils.PATTERN_DATE).equalsIgnoreCase(filterdate)) continue;

            //
            if(msgid.equalsIgnoreCase(report.getMsgid()) && mobile.equalsIgnoreCase(report.getMobile()))
            {
                return report.getStatus() + ":" + report.getReportTime();
            }
        }

        return null;
    }

    /**
     * 查询短信上行
     * @param mobile 手机号码
     * @param begintime 开始时间
     * @param endtime 结束时间
     * @return 上行短信列表(时间:上行号码:消息)
     */
    @Override
    public List<String> mo(String mobile, Date begintime, Date endtime)
    {
        if(StringUtils.IsEmpty(mobile)
            || begintime==null || endtime==null) throw new ZuvException("参数错误");

        Map<String, Object> data = new HashMap<>();
        data.put("account", account);
        data.put("pswd", password);
        data.put("resptype", "json");

        IHttpRes response = OkHttpConn.conn(getreportUrl).data(data).post();
        if(!response.success())
        {
            log.error("request error {} {}", response.status(), response.message());
            throw new ZuvException("request error" + response.message());
        }

        String resstr = response.string();
        log.debug("resstr {}", resstr);
        Reports result = GsonParser.fromJson(resstr, Reports.class);
        log.debug("result {}", result);

        if(!result.success() || result.getMo()==null) return null;

        List<String> list = new ArrayList<>();
        for(Mo mo : result.getMo())
        {
            //过滤时间
            String motime = mo.getMoTime();
            if(StringUtils.IsEmpty(motime)) continue;

            Date modate = DateUtils.format(motime, DATE_PATTERN);
            if(modate.before(begintime) || modate.after(endtime)) continue;

            //没有处理长短信
            if( "1".equalsIgnoreCase(mo.getIsems())) continue;

            //
            if(mobile.equalsIgnoreCase(mo.getMobile()))
            {
                list.add(mo.getMoTime() + ":" + mo.getDestcode() + ":" + mo.getMsg());
            }
        }
        return list;
    }

    //-----------------------------------------------------------------------------------------

    @Override
    public boolean receive_report(String msgid, String status, String mobile, Date stamp)
    {
        log.info("receive_report {} {} {} {}", msgid, status, mobile, stamp);
        return true;
    }

    @Override
    public boolean receive_mo(String message, String dstcode, String mobile, Date stamp)
    {
        log.info("receive_mo {} {} {} {}", message, dstcode, mobile, stamp);
        return true;
    }

    //-----------------------------------------------------------------------------------------

    @Setter
    @Getter
    @ToString
    public class Result
    {
        private long ts; //响应时间
        private int result; //(0表示成功,其他表示失败,具体见定义)
        private int suc; //成功数目
        private int fail; //失败数目
        private String msgid; //为状态匹配使用(如果响应的状态不是0，或者提交时needstatus不等于true，则没有msgid字段)

        public boolean success()
        {
            return result==0;
        }
    }

    @Setter
    @Getter
    @ToString
    public class Balance
    {
        private long ts; //响应时间
        private int result; //(0表示成功,其他表示失败,具体见定义)
        private List<Product> products;

        public boolean success()
        {
            return result==0;
        }
    }

    @Setter
    @Getter
    @ToString
    public class Product
    {
        private String product; //产品ID
        private int num; //条数
    }

    @Setter
    @Getter
    @ToString
    public class Reports
    {
        private int result; //响应状态(0表示成功,其他表示失败,具体见定义)
        private List<Report> report;
        private List<Mo> mo;

        public boolean success()
        {
            return result==0;
        }
    }

    @Setter
    @Getter
    @ToString
    public class Report
    {
        private String msgid;
        private String status; //见定义
        private String mobile;
        private String reportTime; //格式YYMMDDhhmm
    }

    @Setter
    @Getter
    @ToString
    public class Mo
    {
        private String destcode; //用户上行的目的号码(通道接入号+提交时的extno)
        private String spCode; //通道接入号
        private String msg; //上行内容
        private String mobile;
        private String moTime; //格式YYMMDDhhmm
        private String isems; //是否为长短信的一部分(1:是，0:不是) 不带该参数，默认为普通短信
        private String emshead; //
        // isems为1时，本参数以ASCII码形式显示长短信的头信息。
        // 用“,”隔开，分为三个部分，第一部分标识该条长短信的ID（该ID为短信中心生成）；第二部分，表明该长短信的总条数（pk_total）；第三部分，该条短信为该长短信的第几条(pk_number)。
        // 例如：234,4,1，该短信的ID为234,该长短信的总长度为4条，1，当前为第一条。
    }

    //-----------------------------------------------------------------------------------------

    /*
    report里status定义:

    MBBLACK	黑名单号码
    NOROUTE	无通道
    ROUTEERR	通道异常
    REJECT	审核驳回
    DISTURB	手机号码发送次数过多
    EMSERR	长短信不完整
    SIGNERR	签名错
    KEYWORD	敏感词
    WHITESMS	短信内容不在白名单中
     */

    /*
    result响应状态值说明
    代码	说明
    0	提交成功
    101	无此用户
    102	密码错
    104	系统忙（因平台侧原因，暂时无法处理提交的短信）
    105	敏感短信（短信内容包含敏感词）
    106	消息长度错（>700或<=0）
    108	手机号码个数错（>50000或<=0）
    109	无发送额度（该用户可用短信条数为0）
    110	不在发送时间内
    112	无此产品，用户没有订购该产品
    113	extno格式错（非数字或者长度不对）
    115	自动审核驳回
    116	签名不合法，未带签名（用户必须带签名的前提下）
    117	IP地址认证错,请求调用的IP地址不是系统登记的IP地址
    118	用户没有相应的发送权限
    119	用户已过期
    120	内容不在白名单模板中
    */

    //-----------------------------------------------------------------------------------------

}
