/*
 * Id$: zuv-cloud:z-service:cc.zuv.service.spider.lucene.SpiderSearcher:20181225151549
 *
 * SpiderSearcher.java
 * Copyright (c) 2002-2020 Luther Inc.
 * http://zuv.cc
 * All rights reserved.
 */
package cc.zuv.service.spider.lucene;

import cc.zuv.service.spider.ISpiderCode;
import cc.zuv.service.spider.ISpiderDocument;
import lombok.extern.slf4j.Slf4j;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.index.DirectoryReader;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.queryparser.classic.ParseException;
import org.apache.lucene.queryparser.classic.QueryParser;
import org.apache.lucene.search.*;
import org.apache.lucene.search.SortField.Type;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FSDirectory;
import org.apache.lucene.util.Version;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

/**
 * File Description
 *
 * @author			Kama Luther
 * @version			0.1
 * @since           0.1
 * @create.date     2014-2-11 下午02:19:48
 * @modify.date     2014-2-11 下午02:19:48
 */
@Slf4j
public class SpiderSearcher implements ISpiderCode
{

    //-----------------------------------------------------------------------------------------

	private File indexDirectory;
	private Analyzer analyzer;
	private String parseKeyword;
	private int hitsCount;

	public SpiderSearcher(String indexerpath, SpiderParser parser)
	{
		//
		indexDirectory = new File(indexerpath);
    	if(!indexDirectory.exists()) indexDirectory.mkdirs();

    	//
		analyzer = parser.getAnalyzer();
	}

    //-----------------------------------------------------------------------------------------

	public String getParseKeyword()
	{
		return parseKeyword;
	}
	public int getHitsCount()
	{
		return hitsCount;
	}

    //-----------------------------------------------------------------------------------------

	public List<ISpiderDocument> search(int catalog, String keyword, long begindate, long enddate, int start, int limit)
	{
		List<ISpiderDocument> list = new ArrayList<ISpiderDocument>();

		try
		{
			//
			parseKeyword = keyword;

			BooleanQuery query = build(catalog, keyword, begindate, enddate);

	        SortField sortfield_id = new SortField(KEY_IDENTITY, Type.STRING, false);//升序
	        SortField sortfiedl_date = new SortField(KEY_MODIFIED, Type.LONG, true);//降序
	        Sort sorter = new Sort(new SortField[]{sortfield_id, sortfiedl_date});

			//
			Directory dir = FSDirectory.open(indexDirectory);
			IndexReader reader = DirectoryReader.open(dir);
			IndexSearcher searcher = new IndexSearcher(reader);

			//
			TopDocs results = searcher.search(query, 1000, sorter);
			int numTotalHits = results.totalHits;
			hitsCount = numTotalHits;

			//
			if(start < 0) start = 0;
			if(limit < 0) limit = 0;
			if(start > hitsCount) start = hitsCount;
			if(start+limit > hitsCount) limit = hitsCount - start;
			if(start < 0) start = 0;
			if(limit < 0) limit = 0;

			if(numTotalHits>0)
			{
				ScoreDoc[] hits = results.scoreDocs;
				for(int i=start; i<start+limit; i++)
				{
					Document doc = searcher.doc(hits[i].doc);
					list.add(disserialize(doc));
				}
			}

			reader.close();
		}
		catch (IOException e)
		{
			log.error("IO Error", e);
		}
		catch (ParseException e)
		{
			log.error("Parse Error", e);
		}

		return list;
	}

    //-----------------------------------------------------------------------------------------

	private BooleanQuery build(int catalog, String keyword, long begindate, long enddate)
			throws ParseException
	{
		BooleanQuery query = new BooleanQuery();

		QueryParser catalogparser = new QueryParser(Version.LUCENE_46, KEY_CATALOG, analyzer);
		Query catalogquery = catalogparser.parse(catalog+"-");
		query.add(catalogquery, BooleanClause.Occur.SHOULD);

		QueryParser titleparser = new QueryParser(Version.LUCENE_46, KEY_TITLE, analyzer);
		Query titlequery = titleparser.parse(keyword);
		query.add(titlequery, BooleanClause.Occur.SHOULD);

		QueryParser contentparser = new QueryParser(Version.LUCENE_46, KEY_CONTENT, analyzer);
		Query contentquery = contentparser.parse(keyword);
		query.add(contentquery, BooleanClause.Occur.MUST);
		parseKeyword = contentquery.toString(KEY_CONTENT);

		if (begindate > 0 && enddate > 0)
		{
			NumericRangeQuery<Long> rangeQuery = NumericRangeQuery.newLongRange(KEY_MODIFIED, begindate, enddate, true, true);
			query.add(rangeQuery, BooleanClause.Occur.MUST);
		}

		return query;
	}

    private ISpiderDocument disserialize(Document doc)
    {
    	SpiderDocument idoc = new SpiderDocument();
    	String[] ids = doc.get(KEY_IDENTITY).split("\\-");
    	idoc.setCatalog(Integer.parseInt(ids[0]));
    	idoc.setIdentity(ids[1]);
    	idoc.setUrler(doc.get(KEY_URLER));
    	idoc.setTitle(doc.get(KEY_TITLE));
    	idoc.setContent(doc.get(KEY_CONTENT));
    	idoc.setModified(Long.parseLong(doc.get(KEY_MODIFIED)));
        return idoc;
    }

    //-----------------------------------------------------------------------------------------

}
