/*
 * Id$: zuv-cloud:z-service:cc.zuv.service.storage.dfs.impl.DfsLocalService:20190101133152
 *
 * DfsLocalService.java
 * Copyright (c) 2002-2020 Luther Inc.
 * http://zuv.cc
 * All rights reserved.
 */

package cc.zuv.service.storage.dfs.local;

import cc.zuv.ZuvException;
import cc.zuv.service.IServiceCode;
import cc.zuv.service.storage.dfs.IDfsService;
import cc.zuv.utility.CodecUtils;
import cc.zuv.utility.MimeUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.io.*;
import java.util.*;

/**
 * 本地存储
 *
 * @author          Kama Luther
 * @version         0.1
 * @since           0.1
 * @create.date     2019-01-01 13:31
 * @modify.date     2019-01-01 13:31
 */
@Slf4j
@Service
public class LocalDfsService implements IDfsService
{

    //-----------------------------------------------------------------------------------------

    @Autowired
    private LocalDfsConfig config;

    //-----------------------------------------------------------------------------------------

    private LocalDfsConfig.Bucket bucket;
    private String mapping;
    private String phypath;

    @Override
    public void setBucketKey(String bucketkey)
    {
        if(!config.getBuckets().containsKey(bucketkey))
        {
            throw new ZuvException("找不到对应的bucket:" + bucketkey);
        }
        bucket = config.getBuckets().get(bucketkey);
        mapping = bucket.getMapping();
        phypath = bucket.getPhypath();
        phypath = phypath.endsWith("/")?phypath:(phypath+"/");
    }

    //-----------------------------------------------------------------------------------------

    private void validate()
    {
        if(bucket==null)
        {
            Map<String, LocalDfsConfig.Bucket> buckets = config.getBuckets();
            if(buckets !=null && buckets.size()>0)
            {
                String bucketkey = null;
                for(String key : buckets.keySet())
                {
                    bucketkey = key;
                    break;
                }

                log.info("未指定bucket,使用默认bucket: {}", bucketkey);
                setBucketKey(bucketkey);
            }
            else
            {
                throw new ZuvException("服务未初始化, 请先指定BucketKey.");
            }
        }
    }

    private void validatefile(File file)
    {
        if(file==null || file.isDirectory())
        {
            throw new ZuvException("文件不存在或者为目录. key:" + (file!=null?file.getAbsolutePath():""));
        }
    }

    private void validatefold(File fold)
    {
        if(fold==null || !fold.isDirectory())
        {
            throw new ZuvException("目录不存在或者不为目录. key:" + (fold!=null?fold.getAbsolutePath():""));
        }
    }

    //-----------------------------------------------------------------------------------------

    @Override
    public void initial()
    {
        log.info("initial");
    }

    @Override
    public void destroy()
    {
        log.info("destroy");
    }

    //-----------------------------------------------------------------------------------------

    @Override
    public boolean upload(String targetpath, File sourcefile, Map<String, String> meta)
    {
        try
        {
            String phytargetpath = phypath(targetpath);
            File targetfile = new File(phytargetpath);

            FileUtils.copyFile(sourcefile, targetfile);
            log.info("upload: {}", targetfile.getAbsolutePath());
            return true;
        }
        catch (IOException e)
        {
            log.error("上传失败 {}", e.getMessage());
            throw new ZuvException("上传失败", e);
        }
    }

    @Override
    public boolean upload(String targetpath, InputStream sourceis, Map<String, String> meta)
    {
        try
        {
            String phytargetpath = phypath(targetpath);
            File targetfile = new File(phytargetpath);

            FileOutputStream targetos = new FileOutputStream(targetfile);
            IOUtils.copy(sourceis, targetos);
            IOUtils.closeQuietly(targetos);
            IOUtils.closeQuietly(sourceis);

            log.info("upload: {}", targetfile.getAbsolutePath());
            return true;
        }
        catch (IOException e)
        {
            log.error("上传失败 {}", e.getMessage());
            throw new ZuvException("上传失败", e);
        }
    }

    //-----------------------------------------------------------------------------------------

    @Override
    public boolean download(String sourcepath, File targetfile)
    {
        try
        {
            String physourcepath = phypath(sourcepath);
            File sourcefile = new File(physourcepath);

            FileUtils.copyFile(sourcefile, targetfile);
            log.info("download: {}", sourcefile.getAbsolutePath());
            return true;
        }
        catch (IOException e)
        {
            log.error("下载失败 {}", e.getMessage());
            throw new ZuvException("下载失败", e);
        }
    }

    @Override
    public boolean download(String sourcepath, OutputStream targetos)
    {
        try
        {
            String physourcepath = phypath(sourcepath);
            File sourcefile = new File(physourcepath);

            FileInputStream sourceis = new FileInputStream(sourcefile);
            IOUtils.copy(sourceis, targetos);
            IOUtils.closeQuietly(targetos);
            IOUtils.closeQuietly(sourceis);

            log.info("download: {}", sourcefile.getAbsolutePath());
            return true;
        }
        catch (IOException e)
        {
            log.error("上传失败 {}", e.getMessage());
            throw new ZuvException("上传失败", e);
        }
    }

    //-----------------------------------------------------------------------------------------

    @Override
    public boolean exist(String sourcepath)
    {
        String physourcepath = phypath(sourcepath);
        return new File(physourcepath).exists();
    }

    //-----------------------------------------------------------------------------------------

    @Override
    public boolean movefile(String sourcepath, String targetpath)
    {
        try
        {
            String physourcepath = phypath(sourcepath);
            File sourcefile = new File(physourcepath);
            validatefile(sourcefile);

            String phytargetpath = phypath(targetpath);
            File targetfile = new File(phytargetpath);

            FileUtils.moveFile(sourcefile, targetfile);
            log.info("movefile: {} {}", sourcefile.getAbsolutePath(), targetfile.getAbsolutePath());
            return true;
        }
        catch (IOException e)
        {
            log.error("移动文件失败 {}", e.getMessage());
            throw new ZuvException("移动文件失败", e);
        }
    }

    @Override
    public boolean copyfold(String sourcefold, String targetfold)
    {
        try
        {
            String physourcepath = phypath(sourcefold);
            File sourcefile = new File(physourcepath);
            validatefold(sourcefile);

            String phytargetpath = phypath(targetfold);
            File targetfile = new File(phytargetpath);

            FileUtils.copyDirectory(sourcefile, targetfile);
            log.info("copyfold: {} {}", sourcefile.getAbsolutePath(), targetfile.getAbsolutePath());
            return true;
        }
        catch (IOException e)
        {
            log.error("复制目录失败 {}", e.getMessage());
            throw new ZuvException("复制目录失败", e);
        }
    }

    @Override
    public boolean copyfile(String sourcepath, String targetpath)
    {
        try
        {
            String physourcepath = phypath(sourcepath);
            File sourcefile = new File(physourcepath);
            validatefile(sourcefile);

            String phytargetpath = phypath(targetpath);
            File targetfile = new File(phytargetpath);

            FileUtils.copyFile(sourcefile, targetfile);
            log.info("copyfile: {} {}", sourcefile.getAbsolutePath(), targetfile.getAbsolutePath());
            return true;
        }
        catch (IOException e)
        {
            log.error("复制文件失败 {}", e.getMessage());
            throw new ZuvException("复制文件失败", e);
        }
    }

    //-----------------------------------------------------------------------------------------

    @Override
    public boolean mkfold(String sourcefold)
    {
        try
        {
            String physourcepath = phypath(sourcefold);
            File sourcefile = new File(physourcepath);

            FileUtils.forceMkdir(sourcefile);
            log.info("mkfold: {}", sourcefile.getAbsolutePath());
            return true;
        }
        catch (IOException e)
        {
            log.error("创建目录失败 {}", e.getMessage());
            throw new ZuvException("创建目录失败", e);
        }
    }

    @Override
    public boolean rmfold(String sourcefold)
    {
        try
        {
            String physourcepath = phypath(sourcefold);
            File sourcefile = new File(physourcepath);
            validatefold(sourcefile);

            FileUtils.deleteDirectory(sourcefile);
            log.info("rmfold: {}", sourcefile.getAbsolutePath());
            return true;
        }
        catch (IOException e)
        {
            log.error("删除目录失败 {}", e.getMessage());
            throw new ZuvException("删除目录失败", e);
        }
    }

    @Override
    public boolean rmfile(String sourcepath)
    {
        String physourcepath = phypath(sourcepath);
        File sourcefile = new File(physourcepath);
        validatefile(sourcefile);

        boolean delete = FileUtils.deleteQuietly(sourcefile);
        log.info("rmfold: {}", sourcefile.getAbsolutePath());
        return delete;
    }

    @Override
    public List<Map<String, Object>> lsfold(String sourcefold)
    {
        String physourcepath = phypath(sourcefold);
        File sourcefile = new File(physourcepath);
        validatefold(sourcefile);

        List<Map<String, Object>> list = new ArrayList<>();
        File[] files = sourcefile.listFiles();
        if(files!=null)
        {
            for(File file : files)
            {
                Map<String, Object> data = new HashMap<>();
                data.put(FILE_META_FILEKEY, file.getName());
                data.put(FILE_META_FILEKIND, sourcefile.isDirectory()? IServiceCode.FILE_KIND_FOLD: IServiceCode.FILE_KIND_FILE);
                list.add(data);
            }
        }
        return list;
    }

    //-----------------------------------------------------------------------------------------

    @Override
    public Map<String, Object> filemeta(String sourcepath)
    {
        String physourcepath = phypath(sourcepath);
        File sourcefile = new File(physourcepath);
        validatefile(sourcefile);

        return get_meta_native(sourcefile);
    }

    @Override
    public String filehash(String sourcepath)
    {
        String physourcepath = phypath(sourcepath);
        File sourcefile = new File(physourcepath);
        validatefile(sourcefile);

        return CodecUtils.md5(sourcefile);
    }

    @Override
    public long filedate(String sourcepath)
    {
        String physourcepath = phypath(sourcepath);
        File sourcefile = new File(physourcepath);
        validatefile(sourcefile);

        return sourcefile.lastModified();
    }

    @Override
    public long filesize(String sourcepath)
    {
        String physourcepath = phypath(sourcepath);
        File sourcefile = new File(physourcepath);
        validatefile(sourcefile);

        return sourcefile.length();
    }

    //-----------------------------------------------------------------------------------------

    @Override
    public String phypath(String path)
    {
        validate();
        return phypath + path;
    }

    @Override
    public String fileuri(String sourcepath)
    {
        validate();
        return mapping + "/" + CodecUtils.url_encode(sourcepath);
    }

    //-----------------------------------------------------------------------------------------

    public Map<String, Object> get_meta_native(File sourcefile)
    {
        if(!sourcefile.exists()) return null;

        String mime = MimeUtils.guessMimeByFileName(sourcefile.getAbsolutePath());
        byte filekind = sourcefile.isDirectory() ? IServiceCode.FILE_KIND_FOLD: IServiceCode.FILE_KIND_FILE;

        Map<String, Object> data = new HashMap<>();
        data.put(FILE_META_FILENAME, sourcefile.getName());
        data.put(FILE_META_FILEHASH, CodecUtils.md5(sourcefile));
        data.put(FILE_META_FILESIZE, sourcefile.length()+"");
        data.put(FILE_META_FILEETAG, sourcefile.getName());
        data.put(FILE_META_FILEDATE, sourcefile.lastModified()+"");
        data.put(FILE_META_FILEMIME, mime);
        data.put(FILE_META_FILEKIND, filekind+"");
        return data;
    }

    public Map<String, String> bld_meta_native(File sourcefile)
    {
        if(!sourcefile.exists()) return null;

        String mime = MimeUtils.guessMimeByFileName(sourcefile.getAbsolutePath());
        byte filekind = sourcefile.isDirectory() ? IServiceCode.FILE_KIND_FOLD: IServiceCode.FILE_KIND_FILE;

        Map<String, String> data = new HashMap<>();
        data.put(FILE_META_FILENAME, sourcefile.getName());
        data.put(FILE_META_FILEHASH, CodecUtils.md5(sourcefile));
        data.put(FILE_META_FILESIZE, sourcefile.length()+"");
        data.put(FILE_META_FILEETAG, sourcefile.getName());
        data.put(FILE_META_FILEDATE, sourcefile.lastModified()+"");
        data.put(FILE_META_FILEMIME, mime);
        data.put(FILE_META_FILEKIND, filekind+"");
        return data;
    }

    //-----------------------------------------------------------------------------------------

}
