/*
 * Id$: zuv-cloud:z-service:cc.zuv.service.storage.ldap.LDAPConnector:20181225151549
 *
 * LDAPConnector.java
 * Copyright (c) 2002-2020 Luther Inc.
 * http://zuv.cc
 * All rights reserved.
 */
package cc.zuv.service.storage.ldap;
import cc.zuv.ZuvException;
import lombok.extern.slf4j.Slf4j;

import java.util.*;
import  javax.naming.*;
import  javax.naming.directory.*;

/**
 * LDAP操作类
 *
 * @author          Kama Luther
 * @version         0.1
 * @since           0.1
 * @create.date     2018-08-17 16:54
 * @modify.date     2018-08-17 16:54
 */
@Slf4j
public class LDAPConnector
{

    //-----------------------------------------------------------------------------------------

    private static final String DEFAULT_FACTORY        = "com.sun.jndi.ldap.LdapCtxFactory";
    private static final String DEFAULT_AUTHENTICATION = "simple";

    //-----------------------------------------------------------------------------------------

    private Hashtable<String, String> env   = new Hashtable<>();		    //操作的环境Env
    private DirContext ctx 			        = null;						    //操作的内容Context

    //创建
    private Map<String, Attribute> crtmap   = new TreeMap<>();              //创建属性

    //修改
    private Vector<ModificationItem> modvec = new Vector<>();				//调整属性

    //搜索
    private SearchControls searchctl        = new SearchControls();         //搜索控制

    //-----------------------------------------------------------------------------------------

    public LDAPConnector(String url, String binddn, String bindpwd)
    {
        env.put(Context.PROVIDER_URL, url);
        env.put(Context.SECURITY_PRINCIPAL, binddn);
        env.put(Context.SECURITY_CREDENTIALS, bindpwd);

        if (!env.containsKey(Context.INITIAL_CONTEXT_FACTORY))
        {
            env.put(Context.INITIAL_CONTEXT_FACTORY, DEFAULT_FACTORY);
        }
        if (env.containsKey(Context.SECURITY_PRINCIPAL) && !env.containsKey(Context.SECURITY_AUTHENTICATION))
        {
            env.put(Context.SECURITY_AUTHENTICATION, DEFAULT_AUTHENTICATION);
        }
    }

    //-----------------------------------------------------------------------------------------

    public LDAPConnector connect()
    {
        try
        {
            ctx = new InitialDirContext(env);
            log.info("连接成功");
        }
        catch (AuthenticationException e)
        {
            log.error("认证失败 {}", e.getMessage());
            throw new ZuvException("语法错误", e);
        }
        catch (NamingException e)
        {
            log.error("连接失败 {}", e.getMessage());
            throw new ZuvException("连接失败", e);
        }
        return this;
    }

    public void disconn()
    {
        try
        {
            if (ctx != null) ctx.close();
            log.info("断开连接成功");
        }
        catch (NamingException e)
        {
            log.error("断开连接失败 {}", e.getMessage());
            throw new ZuvException("断开连接失败", e);
        }
        finally
        {
            ctx = null;
        }
    }

    //-----------------------------------------------------------------------------------------

    public boolean isConnected()
    {
        return (ctx != null);
    }

    public DirContext getContext()
    {
        return ctx;
    }

    public Hashtable getEnv()
    {
        return env;
    }

    public void setProperty(String key, String val)
    {
        env.put(key, val);
    }

    public String getProperty(String key)
    {
        return env.get(key);
    }

    //-----------------------------------------------------------------------------------------

    /*
     * 设置增加实体的属性。可以多次调用。
     */
    public void setCreateAttr(String key, Object val)
    {
        if(crtmap.containsKey(key))
        {
            Attribute attr = crtmap.get(key);
            attr.add(val);
        }
        else
        {
            Attribute attr = new BasicAttribute(key, val);
            crtmap.put(key, attr);
        }
    }

    /*
     * 增加实体。在此之前需要设置增加实体的属性。
     */
    public void createEntry(String dn)
    {
        //
        Attributes attributes = new BasicAttributes(true);
        for(String key : crtmap.keySet())
        {
            attributes.put(crtmap.get(key));
        }

        //
        try
        {
            ctx.createSubcontext(dn, attributes);
        }
        catch (NamingException e)
        {
            log.error("添加失败 {}", e.getMessage());
            throw new ZuvException("添加失败", e);
        }
    }

    /*
     * 增加实体
     */
    public void bindEntry(String dn, Object obj)
    {
        //
        try
        {
            ctx.bind(dn, obj);
        }
        catch (NamingException e)
        {
            log.error("绑定错误 {}", e.getMessage());
            throw new ZuvException("绑定错误", e);
        }
    }

    /*
     * 取消绑定实体。
     */
    public void unbindEntry(String dn)
    {
        //
        try
        {
            ctx.unbind(dn);
        }
        catch (NamingException e)
        {
            log.error("绑定错误 {}", e.getMessage());
            throw new ZuvException("绑定错误", e);
        }
    }

    /*
     * 删除实体
     */
    public void deleteEntry(String dn)
    {
        //
        try
        {
            ctx.destroySubcontext(dn);
        }
        catch (NamingException e)
        {
            log.error("删除错误 {}", e.getMessage());
            throw new ZuvException("删除错误", e);
        }
    }

    //-----------------------------------------------------------------------------------------

    /*
     * 设置调整的属性 支持增加多个属性
     */
    public void setModifyAttr(String key, Object val, int operationType)
    {
        if (operationType != 0)
        {
            Attribute attr = new BasicAttribute(key, val);
            modvec.add(new ModificationItem(operationType, attr));
        }
    }

    /*
     * 调整属性
     */
    public void modifyEntry(String dn)
    {
        //
        Enumeration en = modvec.elements();
        ModificationItem[] item = new ModificationItem[modvec.size()];
        for (int i = 0; en.hasMoreElements(); i++)
        {
            item[i] = (ModificationItem) en.nextElement();
        }

        //
        try
        {
            ctx.modifyAttributes(dn, item);
        }
        catch (NamingException e)
        {
            log.error("修改失败 {}", e.getMessage());
            throw new ZuvException("修改失败", e);
        }
    }

    //-----------------------------------------------------------------------------------------

    public void setSearchScope(String scope)
    {
        if (scope == null || scope.equalsIgnoreCase("subtree"))
            searchctl.setSearchScope(SearchControls.SUBTREE_SCOPE);
        else if (scope.equalsIgnoreCase("onelevel"))
            searchctl.setSearchScope(SearchControls.ONELEVEL_SCOPE);
        else if (scope.equalsIgnoreCase("object"))
            searchctl.setSearchScope(SearchControls.OBJECT_SCOPE);
    }

    public void setSearchLimit(int limit)
    {
        searchctl.setCountLimit(limit);
    }

    public void searchEntry(String dn, String filter)
    {
        try
        {
            results = ctx.search(dn, filter, searchctl);
        }
        catch (NamingException e)
        {
            log.error("搜索错误 {}", e.getMessage());
            throw new ZuvException("搜索错误", e);
        }
    }

    private NamingEnumeration<SearchResult> results;
    private SearchResult result;
    private String dn;
    private Attributes attributes;

    public boolean next()
    {
        if (results.hasMoreElements())
        {
            try
            {
                result = results.next();
            }
            catch (NamingException e)
            {
                log.error("加载错误 {}", e.getMessage());
                throw new ZuvException("加载错误", e);
            }
            dn = result.getName();
            attributes = result.getAttributes();
            return true;
        }
        return false;
    }

    public Attribute getAttribute(String attrname)
    {
        return attributes.get(attrname);
    }

    public String getStringAttribute(String attrname)
    {
        try
        {
            Attribute attr = attributes.get(attrname);
            return attr != null ? (String) attr.get() : "";
        }
        catch (NamingException e)
        {
            log.error("加载错误 {}", e.getMessage());
            throw new ZuvException("加载错误", e);
        }
    }

    public String getStringAttribute(String attrname, int idx)
    {
        try
        {
            Attribute attr = attributes.get(attrname);
            return attr != null ? (String) attr.get(idx) : "";
        }
        catch (NamingException e)
        {
            log.error("加载错误 {}", e.getMessage());
            throw new ZuvException("加载错误", e);
        }
    }

    public int getAttributeSize(String attrname)
    {
        Attribute attr = attributes.get(attrname);
        return attr != null ? attr.size() : 0;
    }

    public String getRunningDn()
    {
        return dn;
    }

    //-----------------------------------------------------------------------------------------

    /*
     * 简单的搜索
     */
    public Object search(String dn, String attr)
    {
        try
        {
            Attributes attrs = ctx.getAttributes(dn);
            return attrs.get(attr).get();
        }
        catch (NamingException e)
        {
            log.error("搜索错误 {}", e.getMessage());
            throw new ZuvException("搜索错误", e);
        }
    }

    //-----------------------------------------------------------------------------------------

}
