/*
 * Id$: zuv-cloud:z-service:cc.zuv.service.wechat.wxcp.config.WechatCpConfig:20190214170102
 *
 * WechatCpConfig.java
 * Copyright (c) 2002-2020 Luther Inc.
 * http://zuv.cc
 * All rights reserved.
 */

package cc.zuv.service.wechat.wxcp.config;

import cc.zuv.service.wechat.wxcp.handler.*;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import me.chanjar.weixin.common.api.WxConsts;
import me.chanjar.weixin.cp.WxCpConsts;
import me.chanjar.weixin.cp.api.WxCpService;
import me.chanjar.weixin.cp.api.impl.WxCpServiceImpl;
import me.chanjar.weixin.cp.config.WxCpInMemoryConfigStorage;
import me.chanjar.weixin.cp.message.WxCpMessageRouter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Configuration;

import javax.annotation.PostConstruct;
import java.util.Map;

/**
 * zuv-cloud File Description
 *
 * @author          Kama Luther
 * @version         0.1
 * @since           0.1
 * @create.date     2019-02-14 17:01
 * @modify.date     2019-02-14 17:01
 */
@Slf4j
@Configuration
@ConditionalOnProperty(name = "wechat.wxcp.enabled", havingValue = "true")
@EnableConfigurationProperties(WechatCpProperties.class)
public class WechatCpConfig
{

    //-----------------------------------------------------------------------------------------

    private static Map<Integer, WxCpMessageRouter> routers = Maps.newHashMap();
    private static Map<Integer, WxCpService> services = Maps.newHashMap();

    public static Map<Integer, WxCpMessageRouter> getRouters()
    {
        return routers;
    }

    public static WxCpService getCpService(Integer agentId)
    {
        WxCpService wxService = services.get(agentId);
        if (wxService == null)
        {
            throw new IllegalArgumentException(String.format("未找到对应agentId=[%s]的配置，请核实！", agentId));
        }
        return wxService;
    }

    //-----------------------------------------------------------------------------------------

    @Autowired
    private McLogHandler mcLogHandler;

    @Autowired
    private McLocationHandler mcLocationHandler;

    @Autowired
    private McMenuHandler mcMenuHandler;

    @Autowired
    private McMsgHandler mcMsgHandler;

    @Autowired
    private McScanHandler mcScanHandler;

    @Autowired
    private McUnsubscribeHandler mcUnsubscribeHandler;

    @Autowired
    private McSubscribeHandler mcSubscribeHandler;

    @Autowired
    private ContactChangeHandler contactChangeHandler;

    @Autowired
    private EnterAgentHandler enterAgentHandler;

    //-----------------------------------------------------------------------------------------

    @Autowired
    private WechatCpProperties properties;

    @PostConstruct
    public void initServices()
    {
        String corpId = properties.getCorpId();
        for(WechatCpProperties.Config c : properties.getConfigs())
        {
            Integer agentId = c.getAgentId();
            WxCpInMemoryConfigStorage config = new WxCpInMemoryConfigStorage();
            config.setAgentId(agentId);
            config.setCorpId(corpId);
            config.setCorpSecret(c.getSecret());
            config.setToken(c.getToken());
            config.setAesKey(c.getAesKey());

            WxCpService service = new WxCpServiceImpl();
            service.setWxCpConfigStorage(config);

            services.put(agentId, service);
            routers.put(agentId, newRouter(service));
        }
        log.info("init services");
    }

    private WxCpMessageRouter newRouter(WxCpService wxCpService)
    {
        final WxCpMessageRouter newRouter = new WxCpMessageRouter(wxCpService);

        // 记录所有事件的日志(异步执行)
        newRouter.rule().handler(mcLogHandler).next();

        //
        newRouter.rule().async(false)
            .msgType(WxConsts.XmlMsgType.EVENT)
            .event(WxCpConsts.EventType.CHANGE_CONTACT)
            .handler(contactChangeHandler)
            .end();

        newRouter.rule().async(false)
            .msgType(WxConsts.XmlMsgType.EVENT)
            .event(WxCpConsts.EventType.ENTER_AGENT)
            .handler(enterAgentHandler)
            .end();

        // 自定义菜单事件(点击交互)
        newRouter.rule().async(false)
            .msgType(WxConsts.XmlMsgType.EVENT)
            .event(WxConsts.MenuButtonType.CLICK)
            .handler(mcMenuHandler)
            .end();

        // 点击菜单链接事件(链接跳转)
        newRouter.rule().async(false)
            .msgType(WxConsts.XmlMsgType.EVENT)
            .event(WxConsts.MenuButtonType.VIEW)
            .handler(mcMenuHandler)
            .end();

        // 关注事件
        newRouter.rule().async(false)
            .msgType(WxConsts.XmlMsgType.EVENT)
            .event(WxConsts.EventType.SUBSCRIBE)
            .handler(mcSubscribeHandler)
            .end();

        // 取消关注事件
        newRouter.rule().async(false)
            .msgType(WxConsts.XmlMsgType.EVENT)
            .event(WxConsts.EventType.UNSUBSCRIBE)
            .handler(mcUnsubscribeHandler)
            .end();

        // 扫码事件
        newRouter.rule().async(false)
            .msgType(WxConsts.XmlMsgType.EVENT)
            .event(WxConsts.EventType.SCAN)
            .handler(mcScanHandler)
            .end();

        // 上报地理位置事件
        newRouter.rule().async(false)
            .msgType(WxConsts.XmlMsgType.EVENT)
            .event(WxConsts.EventType.LOCATION)
            .handler(mcLocationHandler)
            .end();

        // 接收地理位置消息
        newRouter.rule().async(false)
            .msgType(WxConsts.XmlMsgType.LOCATION)
            .handler(mcLocationHandler)
            .end();

        // 默认
        newRouter.rule().async(false).handler(mcMsgHandler).end();

        return newRouter;
    }

    //-----------------------------------------------------------------------------------------




}
