/*
 * Id$: zuv-cloud:z-service:cc.zuv.service.wechat.wxmp.config.WechatMpConfig:20190212152547
 *
 * WechatMpConfig.java
 * Copyright (c) 2002-2020 Luther Inc.
 * http://zuv.cc
 * All rights reserved.
 */

package cc.zuv.service.wechat.wxmp.config;

import cc.zuv.service.wechat.wxmp.handler.*;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import me.chanjar.weixin.mp.api.WxMpInMemoryConfigStorage;
import me.chanjar.weixin.mp.api.WxMpMessageRouter;
import me.chanjar.weixin.mp.api.WxMpService;
import me.chanjar.weixin.mp.api.impl.WxMpServiceImpl;
import me.chanjar.weixin.mp.constant.WxMpEventConstants;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Configuration;

import javax.annotation.PostConstruct;

import static me.chanjar.weixin.common.api.WxConsts.*;

import java.util.Map;

/**
 * zuv-cloud File Description
 *
 * @author          Kama Luther
 * @version         0.1
 * @since           0.1
 * @create.date     2019-02-12 15:25
 * @modify.date     2019-02-12 15:25
 */
@Slf4j
@Configuration
@ConditionalOnProperty(name = "wechat.wxmp.enabled", havingValue = "true")
@EnableConfigurationProperties(WechatMpProperties.class)
public class WechatMpConfig
{

    //-----------------------------------------------------------------------------------------

    private static Map<String, WxMpMessageRouter> routers = Maps.newHashMap();
    private static Map<String, WxMpService> services = Maps.newHashMap();

    public static Map<String, WxMpMessageRouter> getRouters()
    {
        return routers;
    }

    public static WxMpService getMpService(String appid)
    {
        WxMpService wxService = services.get(appid);
        if (wxService == null)
        {
            throw new IllegalArgumentException(String.format("未找到对应appid=[%s]的配置，请核实", appid));
        }
        return wxService;
    }

    //-----------------------------------------------------------------------------------------

    @Autowired
    private MpLogHandler mpLogHandler;

    @Autowired
    private MpLocationHandler mpLocationHandler;

    @Autowired
    private MpMenuHandler mpMenuHandler;

    @Autowired
    private MpMsgHandler mpMsgHandler;

    @Autowired
    private MpUnsubscribeHandler mpUnsubscribeHandler;

    @Autowired
    private MpSubscribeHandler mpSubscribeHandler;

    @Autowired
    private MpScanHandler mpScanHandler;

    @Autowired
    private KfSessionHandler kfSessionHandler;

    @Autowired
    private StoreCheckNotifyHandler storeCheckNotifyHandler;

    //-----------------------------------------------------------------------------------------

    @Autowired
    private WechatMpProperties properties;


    @PostConstruct
    public void initServices()
    {
        for(WechatMpProperties.Config c : properties.getConfigs())
        {
            String appid = c.getAppId();
            WxMpInMemoryConfigStorage config = new WxMpInMemoryConfigStorage();
            config.setAppId(appid);
            config.setSecret(c.getSecret());
            config.setToken(c.getToken());
            config.setAesKey(c.getAesKey());

            WxMpService service = new WxMpServiceImpl();
            service.setWxMpConfigStorage(config);

            services.put(appid, service);
            routers.put(appid, newRouter(service));
        }
        log.info("init services");
    }

    private WxMpMessageRouter newRouter(WxMpService wxMpService)
    {
        final WxMpMessageRouter newRouter = new WxMpMessageRouter(wxMpService);

        // 记录所有事件的日志(异步执行)
        newRouter.rule().handler(mpLogHandler).next();

        // 接收客服会话管理事件
        newRouter.rule().async(false)
            .msgType(XmlMsgType.EVENT)
            .event(WxMpEventConstants.CustomerService.KF_CREATE_SESSION)
            .handler(kfSessionHandler)
            .end();
        newRouter.rule().async(false)
            .msgType(XmlMsgType.EVENT)
            .event(WxMpEventConstants.CustomerService.KF_CLOSE_SESSION)
            .handler(kfSessionHandler)
            .end();
        newRouter.rule().async(false)
            .msgType(XmlMsgType.EVENT)
            .event(WxMpEventConstants.CustomerService.KF_SWITCH_SESSION)
            .handler(kfSessionHandler)
            .end();

        // 门店审核事件
        newRouter.rule().async(false)
            .msgType(XmlMsgType.EVENT)
            .event(WxMpEventConstants.POI_CHECK_NOTIFY)
            .handler(storeCheckNotifyHandler)
            .end();

        // 自定义菜单事件(点击交互)
        newRouter.rule().async(false)
            .msgType(XmlMsgType.EVENT)
            .event(MenuButtonType.CLICK)
            .handler(mpMenuHandler)
            .end();

        // 点击菜单连接事件(链接跳转)
        newRouter.rule().async(false)
            .msgType(XmlMsgType.EVENT)
            .event(MenuButtonType.VIEW)
            .handler(mpMenuHandler)
            .end();

        // 关注事件
        newRouter.rule().async(false)
            .msgType(XmlMsgType.EVENT)
            .event(EventType.SUBSCRIBE)
            .handler(mpSubscribeHandler)
            .end();

        // 取消关注事件
        newRouter.rule().async(false)
            .msgType(XmlMsgType.EVENT)
            .event(EventType.UNSUBSCRIBE)
            .handler(mpUnsubscribeHandler)
            .end();

        // 扫码事件
        newRouter.rule().async(false)
            .msgType(XmlMsgType.EVENT)
            .event(EventType.SCAN)
            .handler(mpScanHandler)
            .end();

        // 上报地理位置事件
        newRouter.rule().async(false)
            .msgType(XmlMsgType.EVENT)
            .event(EventType.LOCATION)
            .handler(mpLocationHandler)
            .end();

        // 接收地理位置消息
        newRouter.rule().async(false)
            .msgType(XmlMsgType.LOCATION)
            .handler(mpLocationHandler)
            .end();

        // 默认
        newRouter.rule().async(false).handler(mpMsgHandler).end();

        return newRouter;
    }

    //-----------------------------------------------------------------------------------------

}
