/*
 * Decompiled with CFR 0.152.
 */
package cloud.metaapi.sdk.meta_api;

import cloud.metaapi.sdk.clients.meta_api.SynchronizationListener;
import cloud.metaapi.sdk.clients.meta_api.models.ConnectionHealthStatus;
import cloud.metaapi.sdk.clients.meta_api.models.MetatraderSession;
import cloud.metaapi.sdk.clients.meta_api.models.MetatraderSessions;
import cloud.metaapi.sdk.clients.meta_api.models.MetatraderSymbolPrice;
import cloud.metaapi.sdk.clients.meta_api.models.MetatraderSymbolSpecification;
import cloud.metaapi.sdk.meta_api.MetaApiConnection;
import cloud.metaapi.sdk.meta_api.reservoir.Reservoir;
import cloud.metaapi.sdk.util.Js;
import cloud.metaapi.sdk.util.ServiceProvider;
import java.lang.reflect.Field;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Timer;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class ConnectionHealthMonitor
extends SynchronizationListener {
    protected static int minMeasureInterval = 1000;
    protected static int minQuoteInterval = 60000;
    private static Logger logger = LogManager.getLogger(ConnectionHealthMonitor.class);
    private MetaApiConnection connection;
    private Map<String, Reservoir> uptimeReservoirs;
    private Date priceUpdatedAt;
    private long offset;
    private boolean quotesHealthy = false;
    private Timer updateMeasurementsInterval;
    private Map<String, SynchronizationListener.HealthStatus> serverHealthStatus = new ConcurrentHashMap<String, SynchronizationListener.HealthStatus>();

    public ConnectionHealthMonitor(MetaApiConnection connection) {
        this.connection = connection;
        this.updateMeasurementsInterval = Js.setTimeout(() -> this.updateMeasurements(), (int)this.getRandomTimeout());
        this.uptimeReservoirs = new ConcurrentHashMap<String, Reservoir>();
        this.uptimeReservoirs.put("5m", new Reservoir(300, 300000));
        this.uptimeReservoirs.put("1h", new Reservoir(600, 3600000));
        this.uptimeReservoirs.put("1d", new Reservoir(1440, 86400000));
        this.uptimeReservoirs.put("1w", new Reservoir(168, 604800000));
    }

    private void updateMeasurements() {
        this.updateQuoteHealthStatus();
        this.measureUptime();
        this.updateMeasurementsInterval = Js.setTimeout(() -> this.updateMeasurements(), (int)this.getRandomTimeout());
    }

    public void stop() {
        this.updateMeasurementsInterval.cancel();
    }

    @Override
    public CompletableFuture<Void> onSymbolPriceUpdated(String instanceIndex, MetatraderSymbolPrice price) {
        try {
            SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS");
            long brokerTimestamp = formatter.parse(price.brokerTime).getTime();
            this.priceUpdatedAt = Date.from(ServiceProvider.getNow());
            this.offset = this.priceUpdatedAt.getTime() - brokerTimestamp;
        }
        catch (ParseException e) {
            logger.error("Failed to update quote streaming health status on price update for account " + this.connection.getAccount().getId(), (Throwable)e);
            e.printStackTrace();
        }
        return CompletableFuture.completedFuture(null);
    }

    @Override
    public CompletableFuture<Void> onHealthStatus(String instanceIndex, SynchronizationListener.HealthStatus status) {
        this.serverHealthStatus.put(instanceIndex, status);
        return CompletableFuture.completedFuture(null);
    }

    @Override
    public CompletableFuture<Void> onDisconnected(String instanceIndex) {
        this.serverHealthStatus.remove(instanceIndex);
        return CompletableFuture.completedFuture(null);
    }

    public Optional<SynchronizationListener.HealthStatus> getServerHealthStatus() {
        SynchronizationListener.HealthStatus result = null;
        for (SynchronizationListener.HealthStatus s : this.serverHealthStatus.values()) {
            if (result == null) {
                result = s;
                continue;
            }
            for (Field field : SynchronizationListener.HealthStatus.class.getFields()) {
                try {
                    field.set(result, field.get(field) != null ? field.get(field) : field.get(result));
                }
                catch (IllegalAccessException | IllegalArgumentException e) {
                    logger.error((Object)e);
                }
            }
        }
        return Optional.ofNullable(result);
    }

    public ConnectionHealthStatus getHealthStatus() {
        String message;
        ConnectionHealthStatus status = new ConnectionHealthStatus(){
            {
                this.connected = ConnectionHealthMonitor.this.connection.getTerminalState().isConnected();
                this.connectedToBroker = ConnectionHealthMonitor.this.connection.getTerminalState().isConnectedToBroker();
                this.quoteStreamingHealthy = ConnectionHealthMonitor.this.quotesHealthy;
                this.isSynchronized = ConnectionHealthMonitor.this.connection.isSynchronized();
            }
        };
        boolean bl = status.healthy = status.connected && status.connectedToBroker && status.quoteStreamingHealthy && status.isSynchronized;
        if (status.healthy) {
            message = "Connection to broker is stable. No health issues detected.";
        } else {
            message = "Connection is not healthy because ";
            LinkedList<String> reasons = new LinkedList<String>();
            if (!status.connected) {
                reasons.add("connection to API server is not established or lost");
            }
            if (!status.connectedToBroker) {
                reasons.add("connection to broker is not established or lost");
            }
            if (!status.isSynchronized) {
                reasons.add("local terminal state is not synchronized to broker");
            }
            if (!status.quoteStreamingHealthy) {
                reasons.add("quotes are not streamed from the broker properly");
            }
            message = message + String.join((CharSequence)" and ", reasons) + ".";
        }
        status.message = message;
        return status;
    }

    public Map<String, Double> getUptime() {
        ConcurrentHashMap<String, Double> uptime = new ConcurrentHashMap<String, Double>();
        for (Map.Entry<String, Reservoir> entry : this.uptimeReservoirs.entrySet()) {
            uptime.put(entry.getKey(), entry.getValue().getStatistics().average);
        }
        return uptime;
    }

    private void measureUptime() {
        try {
            for (Reservoir r : this.uptimeReservoirs.values()) {
                r.pushMeasurement(this.connection.getTerminalState().isConnected() && this.connection.getTerminalState().isConnectedToBroker() && this.connection.isSynchronized() && this.quotesHealthy ? 100L : 0L);
            }
        }
        catch (Throwable e) {
            logger.error("Failed to measure uptime for account " + this.connection.getAccount().getId(), e);
        }
    }

    private void updateQuoteHealthStatus() {
        try {
            Date serverDateTime = Date.from(ServiceProvider.getNow().minusMillis(this.offset));
            SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss.SSS");
            String serverTime = formatter.format(serverDateTime);
            Calendar calendar = Calendar.getInstance();
            calendar.setTime(serverDateTime);
            int dayOfWeek = calendar.get(7);
            boolean inQuoteSession = false;
            if (this.priceUpdatedAt == null) {
                this.priceUpdatedAt = Date.from(Instant.now());
            }
            if (this.connection.getSubscribedSymbols().size() == 0) {
                this.priceUpdatedAt = Date.from(Instant.now());
            }
            for (String symbol : this.connection.getSubscribedSymbols()) {
                Optional<MetatraderSymbolSpecification> specification = this.connection.getTerminalState().getSpecification(symbol);
                if (!specification.isPresent()) continue;
                List<MetatraderSession> quoteSessions = this.getQuoteSessions(specification.get().quoteSessions, dayOfWeek);
                for (MetatraderSession session : quoteSessions) {
                    if (session.from.compareTo(serverTime) != -1 && session.from.compareTo(serverTime) != 0 || session.to.compareTo(serverTime) != 1 && session.to.compareTo(serverTime) != 0) continue;
                    inQuoteSession = true;
                }
            }
            this.quotesHealthy = this.connection.getSubscribedSymbols().size() == 0 || !inQuoteSession || ServiceProvider.getNow().toEpochMilli() - this.priceUpdatedAt.getTime() < (long)minQuoteInterval;
        }
        catch (Throwable e) {
            logger.error("Failed to update quote streaming health status for account " + this.connection.getAccount().getId(), e);
        }
    }

    private List<MetatraderSession> getQuoteSessions(MetatraderSessions quoteSessions, int dayOfWeek) {
        ArrayList result = null;
        switch (dayOfWeek) {
            case 1: {
                result = quoteSessions.SUNDAY;
                break;
            }
            case 2: {
                result = quoteSessions.MONDAY;
                break;
            }
            case 3: {
                result = quoteSessions.TUESDAY;
                break;
            }
            case 4: {
                result = quoteSessions.WEDNESDAY;
                break;
            }
            case 5: {
                result = quoteSessions.THURSDAY;
                break;
            }
            case 6: {
                result = quoteSessions.FRIDAY;
                break;
            }
            case 7: {
                result = quoteSessions.SATURDAY;
            }
        }
        return result != null ? result : new ArrayList();
    }

    protected int getRandomTimeout() {
        return (int)(ServiceProvider.getRandom() * 59.0 * 1000.0 + (double)minMeasureInterval);
    }
}

