package cn.allbs.websocket.handler;

import cn.allbs.common.constant.AllbsCoreConstants;
import cn.allbs.common.constant.WebsocketConstant;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.http.HttpUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.server.ServerHttpRequest;
import org.springframework.http.server.ServerHttpResponse;
import org.springframework.http.server.ServletServerHttpRequest;
import org.springframework.web.socket.WebSocketHandler;
import org.springframework.web.socket.server.support.HttpSessionHandshakeInterceptor;

import java.nio.charset.Charset;
import java.util.Map;

/**
 * 自定义拦截器
 * <p>
 * 获得 accessToken 请求参数，设置到 attributes 中
 *
 * @author ChenQi
 */
@Slf4j
public class AllbsWebSocketShakeInterceptor extends HttpSessionHandshakeInterceptor {

    /**
     * 握手前
     *
     * @param request    ServerHttpRequest
     * @param response   ServerHttpResponse
     * @param wsHandler  WebSocketHandler
     * @param attributes attributes
     * @return 是否连接
     */
    @Override
    public boolean beforeHandshake(ServerHttpRequest request, ServerHttpResponse response,
                                   WebSocketHandler wsHandler, Map<String, Object> attributes) throws Exception {
        log.debug(WebsocketConstant.HANDSHAKE_BEGIN);
        // 获得 accessToken
        if (request instanceof ServletServerHttpRequest) {
            // 获得请求参数
            Map<String, String> paramMap = HttpUtil.decodeParamMap(request.getURI().getQuery(), Charset.defaultCharset());
            String accessToken = paramMap.get(AllbsCoreConstants.ACCESS_TOKEN);
            String userName = paramMap.get(AllbsCoreConstants.USERNAME);
            if (StrUtil.isAllNotEmpty(accessToken, userName)) {
                attributes.put(AllbsCoreConstants.ACCESS_TOKEN, accessToken);
                attributes.put(AllbsCoreConstants.USERNAME, userName);
                log.debug(DateUtil.now() + "地址：" + request.getRemoteAddress() + " accessToken: " + accessToken + " 的用户" + userName + "握手成功！");
                log.debug(WebsocketConstant.CONNECT_SUCCESS);
                return true;
            }
        }
        log.debug(WebsocketConstant.CONNECT_FAILED);
        return false;
    }

    /**
     * 握手后
     *
     * @param request   request
     * @param response  response
     * @param wsHandler WebSocketHandler
     * @param exception Exception
     */
    @Override
    public void afterHandshake(ServerHttpRequest request, ServerHttpResponse response, WebSocketHandler wsHandler, Exception exception) {
        log.debug(WebsocketConstant.HANDSHAKE_END);
    }
}
