package cn.antcore.resources.core;

import cn.antcore.resources.utils.Args;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import java.util.Properties;

/**
 * Created by Hong on 2017/12/27.
 */
public class YamlProperties extends Hashtable<Object, Object> {

    private final static String ANNOTATION = "#";

    /**
     * 创建没有默认资源的空资源
     */
    public YamlProperties() {
        super();
    }

    /***
     * 创建有默认资源的非空资源
     * @param defaults  默认
     */
    public YamlProperties(Properties defaults) {
        super(defaults);
    }

    public synchronized Object setProperty(String key, String value) {
        return super.put(key, value);
    }

    public synchronized void load(InputStream inStream) throws IOException {
        read(new LineReader(inStream));
    }

    private void read(LineReader lr) throws IOException {
        String line;
        List<Node> nodes = new ArrayList<Node>();
        Node tempNode = null;
        while ((line = lr.readLine()) != null) {
            //去除单行注释
            line = delAnnotation(line);
            //不处理空行
            if (line.length() == 0) {
                continue;
            }
            //是否是新节点
            if (isNewNode(Args.value(line))) {
                tempNode = readNode(line);
                nodes.add(tempNode);
                continue;
            }
            //读取节点
            Node node = readNode(line);

            if (tempNode.getLevel() > node.getLevel()) {
                //上N级
                int upLevel = tempNode.getLevel() - node.getLevel();
                do {
                    tempNode = tempNode.getPreNode();
                } while ((upLevel-- == 0));
            }
            if (node.getLevel() == tempNode.getLevel()) {
                //平级
                tempNode = tempNode.getPreNode();
            }
            tempNode.addNode(node);
            if (node.getLevel() - 1 == tempNode.getLevel()) {
                //下一级
                tempNode.addNode(node);
                tempNode = node;
            }
        }
        for (Node node : nodes) {
            put(node.getAllName(), node.getValue());
            putNode(node);
        }
    }

    /**
     * 递归添加节点
     */
    private void putNode(Node node) {
        if (node.getNodes() != null) {
            for (Node temp : node.getNodes()) {
                while (temp != null) {
                    put(temp.getAllName(), temp.getValue());
                    if (temp.getNodes() == null) {
                        break;
                    }
                    putNode(temp);
                    break;
                }
            }
        }
    }

    /**
     * 添加资源
     *
     * @param key
     * @param value
     */
    private void put(String key, String value) {
        if (Args.valueTrim(value).length() > 0) {
            super.put(key, value);
        }
    }

    /**
     * 判断是否是新的节点
     *
     * @param str 字符串
     * @return
     */
    private boolean isNewNode(String str) {
        return !str.startsWith(" ");
    }

    /**
     * 读取节点
     *
     * @param str 字符串
     * @return
     */
    private Node readNode(String str) {
        int index = str.indexOf(":");
        String[] values = str.split(":");
        if (values.length >= 2 && index > -1) {
            return new Node(str.substring(0, index), str.substring(index + 1, str.length()));
        }
        return new Node(str.substring(0, index), "");
    }

    /**
     * 删除注释内容
     *
     * @param str 原字符串
     * @return 去除注释后的字符串
     */
    private String delAnnotation(String str) {
        if (str.indexOf(ANNOTATION) > -1) {
            return str.substring(0, str.indexOf(ANNOTATION));
        }
        return str;
    }

    /**
     * 节点
     */
    static class Node {

        private String name;
        private String value;
        private int level = 0;
        private String allName;
        private Node preNode;
        private List<Node> nodes;

        public Node(String name, String value) {
            level(name);
            this.name = Args.valueTrim(name);
            this.value = value;
            this.allName = this.name;
        }

        public String getName() {
            return Args.valueTrim(name);
        }

        public String getValue() {
            return Args.valueTrim(value);
        }

        public int getLevel() {
            return level;
        }

        public String getAllName() {
            return allName;
        }

        public void setAllName(String allName) {
            this.allName = allName;
        }

        public Node getPreNode() {
            return preNode;
        }

        public List<Node> getNodes() {
            return nodes;
        }

        public void addNode(Node node) {
            if (this.nodes == null) {
                this.nodes = new ArrayList<Node>();
            }
            node.preNode = this;
            node.setAllName(this.allName + "." + node.getName());
            this.nodes.add(node);
        }

        public void level(String name) {
            int level = 0;
            char[] chars = name.toCharArray();
            char temp;
            for (char c : chars) {
                temp = c;
                if (temp != ' ') {
                    break;
                }
                level++;
            }
            this.level = level / 2;
        }
    }

    /**
     * 行阅读器
     */
    class LineReader {

        private BufferedReader br;

        public LineReader(InputStream inStream) {
            br = new BufferedReader(new InputStreamReader(inStream));
        }

        public String readLine() throws IOException {
            String line;
            while ((line = br.readLine()) != null) {
                return line;
            }
            if (br != null) {
                br.close();
            }
            return null;
        }
    }


}
