package cn.entertech.log.local

import android.content.Context
import android.util.Log
import cn.android.file.FileHelper
import cn.entertech.base.util.DateUtils
import cn.entertech.log.api.ILogPrinter
import cn.entertech.log.api.LogLevelEnum
import java.io.File

/**
 * 日志信息由 tag+msg 组成
 * @param logFilePath 日志存放文件路径
 * @param logFileName 日志文件名
 * @param dataPrefix 日志前缀 tag之前的部分，比如时间
 * @param dataMidfix 日志中 tag与msg之间部分
 * @param dataSuffix 日志信息后缀 msg之后的部分
 * @param needShowSystemLog 是否需要显示系统日志
 * @param needWriteLog 某个等级的日志是否需要写入文件
 * */
class StorageLogPrinter(
    private val context: Context,
    private val logFilePath: String = context.applicationContext.getExternalFilesDir("log")
        .toString(),
    private val logFileName: String = "${
        DateUtils.getFormatTime(
            System.currentTimeMillis(), "yyyy-MM-dd_HH:mm:ss"
        )
    }.txt",
    private val dataPrefix: (LogLevelEnum) -> String = { "" },
    private val dataMidfix: (LogLevelEnum) -> String = { "" },
    private val dataSuffix: (LogLevelEnum) -> String = { "" },
    private val needShowSystemLog: (LogLevelEnum) -> Boolean = { true },
    private val needWriteLog: (LogLevelEnum) -> Boolean = { true }
) : ILogPrinter {

    private val fileHelper by lazy {
        val fileHelper = FileHelper()
        fileHelper.setFilePath(
            logFilePath, logFileName
        )
        fileHelper
    }

    fun getCurrentLogFile(): File {
        return File(logFilePath, logFileName)
    }

    fun getCurrentLogFolder(): File {
        return File(logFilePath)
    }

    override fun d(tag: String, msg: String, tr: Throwable?) {
        if (needShowSystemLog(LogLevelEnum.DEBUG)) {
            Log.d(tag, msg, tr)
        }
        writeData(LogLevelEnum.DEBUG, tag, msg, needWriteLog)
    }

    override fun i(tag: String, msg: String, tr: Throwable?) {
        if (needShowSystemLog(LogLevelEnum.INFO)) {
            Log.i(tag, msg, tr)
        }
        writeData(LogLevelEnum.INFO, tag, msg, needWriteLog)
    }

    override fun e(tag: String, msg: String, tr: Throwable?) {
        if (needShowSystemLog(LogLevelEnum.ERROR)) {
            Log.e(tag, msg, tr)
        }
        writeData(LogLevelEnum.ERROR, tag, msg, needWriteLog)
    }

    override fun wtf(tag: String, msg: String, tr: Throwable?) {
        if (needShowSystemLog(LogLevelEnum.WHAT_TERRIBLE_FAILURE)) {
            Log.wtf(tag, msg, tr)
        }
        writeData(
            LogLevelEnum.WHAT_TERRIBLE_FAILURE, tag, msg, needWriteLog
        )
    }

    override fun w(tag: String, msg: String, tr: Throwable?) {
        if (needShowSystemLog(LogLevelEnum.WARN)) {
            Log.w(tag, msg, tr)
        }
        writeData(LogLevelEnum.WARN, tag, msg, needWriteLog)
    }

    override fun v(tag: String, msg: String, tr: Throwable?) {
        if (needShowSystemLog(LogLevelEnum.VERBOSE)) {
            Log.v(tag, msg, tr)
        }
        writeData(LogLevelEnum.VERBOSE, tag, msg, needWriteLog)
    }

    fun writeData(
        logLevel: LogLevelEnum,
        tag: String,
        msg: String,
        needWriteLog: (LogLevelEnum) -> Boolean,
    ) {
        if (needWriteLog(logLevel)) {
            fileHelper.writeData(
                "${dataPrefix(logLevel)} $tag ${dataMidfix(logLevel)} $msg ${
                    dataSuffix(
                        logLevel
                    )
                } \n"
            )
        }
    }
}