package io.polaris.core.compress;

import io.polaris.core.io.IO;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

/**
 * GZIP是用于Unix系统的文件压缩<br>
 * gzip的基础是DEFLATE
 *
 * @author Qt
 * @since 1.8
 */
public class Gzip implements Closeable {

	private InputStream source;
	private OutputStream target;

	/**
	 * 创建Gzip
	 *
	 * @param source 源流
	 * @param target 目标流
	 * @return Gzip
	 */
	public static Gzip of(InputStream source, OutputStream target) {
		return new Gzip(source, target);
	}

	/**
	 * 构造
	 *
	 * @param source 源流
	 * @param target 目标流
	 */
	public Gzip(InputStream source, OutputStream target) {
		this.source = source;
		this.target = target;
	}

	/**
	 * 获取目标流
	 *
	 * @return 目标流
	 */
	public OutputStream getTarget() {
		return this.target;
	}

	/**
	 * 将普通数据流压缩
	 *
	 * @return Gzip
	 */
	public Gzip gzip() throws IOException {
		target = (target instanceof GZIPOutputStream) ?
			(GZIPOutputStream) target : new GZIPOutputStream(target);
		IO.copy(source, target);
		((GZIPOutputStream) target).finish();
		return this;
	}

	/**
	 * 将压缩流解压到target中
	 *
	 * @return Gzip
	 */
	public Gzip unGzip() throws IOException {
		source = (source instanceof GZIPInputStream) ?
			(GZIPInputStream) source : new GZIPInputStream(source);
		IO.copy(source, target);
		return this;
	}

	@Override
	public void close() {
		IO.close(this.target);
		IO.close(this.source);
	}
}
