/*
 * Decompiled with CFR 0.152.
 */
package io.polaris.core.crypto.otp;

import io.polaris.core.codec.Base32;
import io.polaris.core.crypto.otp.GoogleAuthenticatorConfig;
import io.polaris.core.crypto.otp.GoogleAuthenticatorException;
import io.polaris.core.crypto.otp.GoogleAuthenticatorKey;
import io.polaris.core.crypto.otp.IGoogleAuthenticator;
import io.polaris.core.crypto.otp.ReseedingSecureRandom;
import io.polaris.core.log.ILogger;
import io.polaris.core.log.ILoggers;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;

public final class GoogleAuthenticator
implements IGoogleAuthenticator {
    private static final ILogger log = ILoggers.of(GoogleAuthenticator.class);
    public static final String RNG_ALGORITHM_KEY = "googleauth.rng.algorithm";
    public static final String RNG_ALGORITHM_PROVIDER_KEY = "googleauth.rng.algorithmProvider";
    private static final String DEFAULT_RANDOM_NUMBER_ALGORITHM = "SHA1PRNG";
    private static final String DEFAULT_RANDOM_NUMBER_ALGORITHM_PROVIDER = "SUN";
    private static final int SCRATCH_CODE_LENGTH = 8;
    public static final int SCRATCH_CODE_MODULUS = (int)Math.pow(10.0, 8.0);
    private static final int SCRATCH_CODE_INVALID = -1;
    private static final int BYTES_PER_SCRATCH_CODE = 4;
    private final GoogleAuthenticatorConfig config;
    private ReseedingSecureRandom secureRandom;

    public GoogleAuthenticator() {
        this.config = new GoogleAuthenticatorConfig();
        this.secureRandom = new ReseedingSecureRandom(this.getRandomNumberAlgorithm(), this.getRandomNumberAlgorithmProvider());
    }

    public GoogleAuthenticator(GoogleAuthenticatorConfig config) {
        if (config == null) {
            throw new IllegalArgumentException("Configuration cannot be null.");
        }
        this.config = config;
        this.secureRandom = new ReseedingSecureRandom(this.getRandomNumberAlgorithm(), this.getRandomNumberAlgorithmProvider());
    }

    public GoogleAuthenticator(String randomNumberAlgorithm, String randomNumberAlgorithmProvider) {
        this(new GoogleAuthenticatorConfig(), randomNumberAlgorithm, randomNumberAlgorithmProvider);
    }

    public GoogleAuthenticator(GoogleAuthenticatorConfig config, String randomNumberAlgorithm, String randomNumberAlgorithmProvider) {
        if (config == null) {
            throw new IllegalArgumentException("Configuration cannot be null.");
        }
        this.config = config;
        if (randomNumberAlgorithm == null && randomNumberAlgorithmProvider == null) {
            this.secureRandom = new ReseedingSecureRandom();
        } else {
            if (randomNumberAlgorithm == null) {
                throw new IllegalArgumentException("RandomNumberAlgorithm must not be null. If the RandomNumberAlgorithm is null, the RandomNumberAlgorithmProvider must also be null.");
            }
            if (randomNumberAlgorithmProvider == null) {
                this.secureRandom = new ReseedingSecureRandom(randomNumberAlgorithm);
            }
        }
    }

    private String getRandomNumberAlgorithm() {
        return System.getProperty(RNG_ALGORITHM_KEY, DEFAULT_RANDOM_NUMBER_ALGORITHM);
    }

    private String getRandomNumberAlgorithmProvider() {
        return System.getProperty(RNG_ALGORITHM_PROVIDER_KEY, DEFAULT_RANDOM_NUMBER_ALGORITHM_PROVIDER);
    }

    int calculateCode(byte[] key, long tm) {
        byte[] data = new byte[8];
        long value = tm;
        int i = 8;
        while (i-- > 0) {
            data[i] = (byte)value;
            value >>>= 8;
        }
        SecretKeySpec signKey = new SecretKeySpec(key, this.config.getHmacHash().getHmacAlgorithm());
        try {
            Mac mac = Mac.getInstance(this.config.getHmacHash().getHmacAlgorithm());
            mac.init(signKey);
            byte[] hash = mac.doFinal(data);
            int offset = hash[hash.length - 1] & 0xF;
            long truncatedHash = 0L;
            for (int i2 = 0; i2 < 4; ++i2) {
                truncatedHash <<= 8;
                truncatedHash |= (long)(hash[offset + i2] & 0xFF);
            }
            truncatedHash &= Integer.MAX_VALUE;
            return (int)(truncatedHash %= (long)this.config.getKeyModulus());
        }
        catch (InvalidKeyException | NoSuchAlgorithmException ex) {
            log.error(ex.getMessage(), ex);
            throw new GoogleAuthenticatorException("The operation cannot be performed now.");
        }
    }

    private long getTimeWindowFromTime(long time) {
        return time / this.config.getTimeStepSizeInMillis();
    }

    private boolean checkCode(String secret, long code, long timestamp, int window) {
        byte[] decodedKey = this.decodeSecret(secret);
        long timeWindow = this.getTimeWindowFromTime(timestamp);
        for (int i = -((window - 1) / 2); i <= window / 2; ++i) {
            long hash = this.calculateCode(decodedKey, timeWindow + (long)i);
            if (hash != code) continue;
            return true;
        }
        return false;
    }

    private byte[] decodeSecret(String secret) {
        return Base32.decode(secret);
    }

    @Override
    public GoogleAuthenticatorKey createCredentials() {
        int bufferSize = this.config.getSecretBits() / 8;
        byte[] buffer = new byte[bufferSize];
        this.secureRandom.nextBytes(buffer);
        byte[] secretKey = Arrays.copyOf(buffer, bufferSize);
        String generatedKey = this.calculateSecretKey(secretKey);
        int validationCode = this.calculateValidationCode(secretKey);
        List<Integer> scratchCodes = this.calculateScratchCodes();
        return new GoogleAuthenticatorKey.Builder(generatedKey).setConfig(this.config).setVerificationCode(validationCode).setScratchCodes(scratchCodes).build();
    }

    @Override
    public GoogleAuthenticatorKey createCredentials(String userName) {
        if (userName == null) {
            throw new IllegalArgumentException("User name cannot be null.");
        }
        GoogleAuthenticatorKey key = this.createCredentials();
        return key;
    }

    private List<Integer> calculateScratchCodes() {
        ArrayList<Integer> scratchCodes = new ArrayList<Integer>();
        for (int i = 0; i < this.config.getNumberOfScratchCodes(); ++i) {
            scratchCodes.add(this.generateScratchCode());
        }
        return scratchCodes;
    }

    private int calculateScratchCode(byte[] scratchCodeBuffer) {
        if (scratchCodeBuffer.length < 4) {
            throw new IllegalArgumentException(String.format("The provided random byte buffer is too small: %d.", scratchCodeBuffer.length));
        }
        int scratchCode = 0;
        for (int i = 0; i < 4; ++i) {
            scratchCode = (scratchCode << 8) + (scratchCodeBuffer[i] & 0xFF);
        }
        if (this.validateScratchCode(scratchCode = (scratchCode & Integer.MAX_VALUE) % SCRATCH_CODE_MODULUS)) {
            return scratchCode;
        }
        return -1;
    }

    boolean validateScratchCode(int scratchCode) {
        return scratchCode >= SCRATCH_CODE_MODULUS / 10;
    }

    private int generateScratchCode() {
        byte[] scratchCodeBuffer;
        int scratchCode;
        do {
            scratchCodeBuffer = new byte[4];
            this.secureRandom.nextBytes(scratchCodeBuffer);
        } while ((scratchCode = this.calculateScratchCode(scratchCodeBuffer)) == -1);
        return scratchCode;
    }

    private int calculateValidationCode(byte[] secretKey) {
        return this.calculateCode(secretKey, 0L);
    }

    @Override
    public int getTotpPassword(String secret) {
        return this.getTotpPassword(secret, System.currentTimeMillis());
    }

    @Override
    public int getTotpPassword(String secret, long time) {
        return this.calculateCode(this.decodeSecret(secret), this.getTimeWindowFromTime(time));
    }

    private String calculateSecretKey(byte[] secretKey) {
        return Base32.encodeToString(secretKey);
    }

    @Override
    public boolean authorize(String secret, int verificationCode) {
        return this.authorize(secret, verificationCode, System.currentTimeMillis());
    }

    @Override
    public boolean authorize(String secret, int verificationCode, long time) {
        if (secret == null) {
            throw new IllegalArgumentException("Secret cannot be null.");
        }
        if (verificationCode <= 0 || verificationCode >= this.config.getKeyModulus()) {
            return false;
        }
        return this.checkCode(secret, verificationCode, time, this.config.getWindowSize());
    }
}

