/**
 * 版权所有(C) 晓祥工作室 2010-2020<br>
 * Copyright 2010-2020 XiaoXiang Workroom.<br>
 * 创建日期 2014-10-7
 */
package bee.cloud.cache;

import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * 功能说明：缓存操作接口
 * <p>
 * 1) String(字符串)操作
 * </p>
 * 可以对单个字符串进行存、取、删除、设置过期日期操作。
 * <p>
 * 2) Hash(哈希表)操作
 * </p>
 * 
 * @author chenxiang 2014-10-7
 */
public interface Cache1
{
    /**
     * 返回 key 所关联的字符串值。
     * 如果 key 不存在那么返回特殊值 null 。
     * 
     * @param key -
     * @return 当 key 不存在时，返回 null ，否则，返回 key 的值。
     * @author 陈祥 2014-10-8
     */
    String get(String key);

    /**
     * 返回 key 所关联的字符串值。
     * 如果 key 不存在那么返回特殊值 null 。
     * 
     * @param keys -
     * @return 当 key 不存在时，返回 null ，否则，返回 key 的值。
     * @author 陈祥 2014-10-8
     */
    Map<String, String> gets(Set<String> keys);

    /**
     * <b>查找所有符合给定模式 pattern 的 key 。</b> <br>
     * KEYS * 匹配数据库中所有 key 。<br>
     * KEYS h?llo 匹配 hello ， hallo 和 hxllo 等。<br>
     * KEYS h*llo 匹配 hllo 和 heeeeello 等。<br>
     * KEYS h[ae]llo 匹配 hello 和 hallo ，但不匹配 hillo 。<br>
     * 特殊符号用 \ 隔开<br>
     * 注意
     * KEYS 的速度非常快，但在一个大的数据库中使用它仍然可能造成性能问题。
     * 
     * @param pattern -
     * @return 符合给定模式的 key 列表。
     * @author 陈祥 2014-10-12
     */
    Set<String> keys(String pattern);

    /**
     * 将字符串值 value 关联到 key 。<br>
     * 如果 key 已经持有其他值， SET 就覆写旧值，无视类型。
     * 
     * @param key -
     * @param value -
     * @author 陈祥 2014-10-8
     */
    void set(String key, String value);

    /**
     * 将值 value 关联到 key ，并将 key 的生存时间设为 seconds (以秒为单位)。<br>
     * 如果 key 已经存在， 将覆写旧值。
     * 
     * @param key
     *            -
     * @param value
     *            -
     * @param seconds
     *            -生存时间，单位（秒）
     * @author 陈祥 2014-10-8
     */
    void set(String key, String value, int seconds);

    /**
     * 删除给定的一个或多个 key，不存在的 key 会被忽略。
     * <p>
     * <b>时间复杂度：</b>
     * </p>
     * O(N)， N 为被删除的 key 的数量。<br>
     * 删除单个字符串类型的 key ，时间复杂度为O(1)。<br>
     * 删除单个列表、集合、有序集合或哈希表类型的 key ，时间复杂度为O(M)， M 为以上数据结构内的元素数量。
     * 
     * @param keys -
     * @author 陈祥 2014-10-8
     */
    void del(String... keys);

    /**
     * 检查给定 key 是否存在。
     * 
     * @param key
     *            -
     * @return 若 key 存在，返回 true ，否则返回 false 。
     * @author 陈祥 2014-10-11
     */
    boolean exists(String key);

    /**
     * 为给定 key 设置生存时间，当 key 过期时(生存时间为 0 )，它会被自动删除。
     * 
     * @param key -
     * @param seconds
     *            -生存时间，单位（秒）
     * @author 陈祥 2014-10-11
     */
    void expire(String key, int seconds);

    /**
     * 为给定 key 设置生存时间，当 key 过期时(生存时间到了指定的日期后 )，它会被自动删除。
     * 
     * @param key -
     * @param date
     *            -生成到期日期
     * @author 陈祥 2014-10-11
     */
    void expire(String key, Date date);

    /**
     * 为给定 key 撤消设置生存的时间
     * 
     * @param key -
     * @author 陈祥 2014-10-11
     */
    void persist(String key);

    /**
     * 将 key 中储存的数字值增一。<br>
     * 如果 key 不存在，那么 key 的值会先被初始化为 0 ，然后再执行 INCR 操作。<br>
     * 如果值包含错误的类型，或字符串类型的值不能表示为数字，那么返回一个错误。<br>
     * 本操作的值限制在 64 位(bit)有符号数字表示之内。<br>
     * 
     * @param key -
     * @return 执行 INCR 命令之后 key 的值。
     * @author 陈祥 2014-10-12
     */
    long incr(String key);

    /**
     * 将 key 所储存的值加上增量 increment 。<br>
     * 如果 key 不存在，那么 key 的值会先被初始化为 0 ，然后再执行 INCRBY 命令。<br>
     * 如果值包含错误的类型，或字符串类型的值不能表示为数字，那么返回一个错误。<br>
     * 本操作的值限制在 64 位(bit)有符号数字表示之内。<br>
     * 
     * @param key -
     * @param increment -
     * @return 加上 increment 之后， key 的值。
     * @author 陈祥 2014-10-12
     */
    long incr(String key, long increment);

    // =====Hash(哈希表)操作

    /**
     * 返回哈希表 key 中的所有域field。
     * 
     * @param key -
     * @return 一个包含哈希表中所有域field的表。<br>
     *         当 key 不存在时，返回一个空表。
     * @author 陈祥 2014-10-12
     */
    Set<String> hkeys(String key);

    /**
     * 查看哈希表 key 中，给定域 field 是否存在。
     * 
     * @param key -
     * @param field -
     * @return 如果哈希表含有给定域，返回 true 。<br>
     *         如果哈希表不含有给定域，或 key 不存在，返回 false 。
     * @author 陈祥 2014-10-12
     */
    boolean hexists(String key, String field);

    /**
     * 返回哈希表 key 中，所有的域field和值。
     * 
     * @param key -
     * @return 以列表形式返回哈希表的域和域的值。<br>
     *         若 key 不存在，返回空Map。
     * @author 陈祥 2014-10-12
     */
    Map<String, String> hgetAll(String key);

    /**
     * 返回哈希表 key 中给定域 field 的值。
     * 
     * @param key -
     * @param field -
     * @return 给定域的值。
     *         当给定域不存在或是给定 key 不存在时，返回 nil 。
     * @author 陈祥 2014-10-12
     */
    String hget(String key, String field);

    /**
     * 返回哈希表 key 中给定域 field 的值。
     * 
     * @param key -
     * @param field -
     * @return 给定域的值集Map。
     *         当给定域不存在或是给定 key 不存在时，返回 nil 。
     * @author 陈祥 2014-10-12
     */
    Map<String, String> hgets(String key, String... field);

    /**
     * 将哈希表 key 中的域 field 的值设为 value 。<br>
     * 如果 key 不存在，一个新的哈希表被创建并进行 HSET 操作。<br>
     * 如果域 field 已经存在于哈希表中，旧值将被覆盖。<br>
     * 
     * @param key -
     * @param values -
     * @author 陈祥 2014-10-12
     */
    void hset(String key, Map<String, String> values);

    /**
     * 将哈希表 key 中的域 field 的值设为 value 。<br>
     * 如果 key 不存在，一个新的哈希表被创建并进行 HSET 操作。<br>
     * 如果域 field 已经存在于哈希表中，旧值将被覆盖。<br>
     * 
     * @param key -
     * @param field -
     * @param value -
     * @author 陈祥 2014-10-12
     */
    void hset(String key, String field, String value);

    /**
     * 删除哈希表 key 中的一个或多个指定域，不存在的域将被忽略。
     * 
     * @param key -
     * @param fields -
     * @author 陈祥 2014-10-12
     */
    void hdel(String key, String... fields);

    /**
     * 为哈希表 key 中的域 field 的值加上增量 increment 。<br>
     * 增量也可以为负数，相当于对给定域进行减法操作。<br>
     * 如果 key 不存在，一个新的哈希表被创建并执行 HINCRBY 命令。<br>
     * 如果域 field 不存在，那么在执行命令前，域的值被初始化为 0 。<br>
     * 对一个储存字符串值的域 field 执行 HINCRBY 命令将造成一个错误。<br>
     * 本操作的值被限制在 64 位(bit)有符号数字表示之内。
     * 
     * @param key -
     * @param field -
     * @param value -
     * @return 执行 HINCRBY 命令之后，哈希表 key 中域 field 的值。
     * @author 陈祥 2014-10-12
     */
    long hincrBy(String key, String field, long value);

    /**
     * 为哈希表 key 中的域 field 的值加1 。<br>
     * 如果 key 不存在，一个新的哈希表被创建并执行 HINCRBY 命令。<br>
     * 如果域 field 不存在，那么在执行命令前，域的值被初始化为 0 。<br>
     * 对一个储存字符串值的域 field 执行 HINCRBY 命令将造成一个错误。<br>
     * 
     * @param key -
     * @param field -
     * @return 执行 HINCRBY 命令之后，哈希表 key 中域 field 的值。
     * @author 陈祥 2014-10-12
     */
    long hincrBy(String key, String field);

    /**
     * 返回哈希表 key 中域的数量。
     * 
     * @param key -
     * @return 哈希表中域的数量。<br>
     *         当 key 不存在时，返回 0 。
     * @author 陈祥 2014-10-12
     */
    long hlen(String key);

    /**
     * 获取一个堆栈
     * 
     * @param key -
     * @return -
     * @author 陈祥 2015-2-11
     */
    Stack getStack(String key);

    /** 列表队列 */
    interface Stack
    {
        /**
         * 将一个或多个值 value 插入到列表 key 的表头(最左边)。
         * 如果有多个 value 值，那么各个 value 值按从左到右的顺序依次插入到表头
         * 
         * @param values -
         * @author 陈祥 2015-2-10
         */
        void lpush(String... values);

        /**
         * 将一个或多个值 value 插入到列表 key 的表尾(最右边)。
         * 如果有多个 value 值，那么各个 value 值按从左到右的顺序依次插入到表尾
         * 
         * @param values -
         * @author 陈祥 2015-2-11
         */
        void rpush(String... values);

        /**
         * 移除并返回列表 key 的头元素。
         * 
         * @return 列表的头元素。
         *         当 key 不存在时，返回 null 。
         * @author 陈祥 2015-2-11
         */
        String lpop();

        /**
         * 移除并返回列表 key 的尾元素。
         * 
         * @return 列表的尾元素。。
         *         当 key 不存在时，返回 null 。
         * @author 陈祥 2015-2-11
         */
        String rpop();

        /**
         * 移除并返回列表 key 的头元素。
         * 
         * @param count
         *            -需要移除的数量
         * @return 列表的头元素。
         *         当 key 不存在时，返回 null 。
         * @author 陈祥 2015-2-11
         */
        List<String> lpop(int count);

        /**
         * 移除并返回列表 key 的尾元素。
         * 
         * @param count
         *            -需要移除的数量
         * @return 列表的尾元素。。
         *         当 key 不存在时，返回 null 。
         * @author 陈祥 2015-2-11
         */
        List<String> rpop(int count);

        /**
         * 返回列表 key 的长度。
         * 如果 key 不存在，则 key 被解释为一个空列表，返回 0 .
         * 如果 key 不是列表类型，返回一个错误。
         * 
         * @return 返回列表 key 的长度。
         * @author 陈祥 2015-2-11
         */
        long llen();

        /**
         * 
         <p>
         * <strong>LRANGE key start stop</strong>
         * </p>
         * <p>
         * 返回列表 <tt class="docutils literal"><span class="pre">key</span></tt> 中指定区间内的元素，区间以偏移量 <tt class="docutils literal"><span class="pre">start</span></tt> 和
         * <tt class="docutils literal"><span class="pre">stop</span></tt> 指定。
         * </p>
         * <p>
         * 下标(index)参数 <tt class="docutils literal"><span class="pre">start</span></tt> 和 <tt class="docutils literal"><span class="pre">stop</span></tt> 都以
         * <tt class="docutils literal"><span class="pre">0</span></tt> 为底，也就是说，以 <tt class="docutils literal"><span class="pre">0</span></tt> 表示列表的第一个元素，以
         * <tt class="docutils literal"><span class="pre">1</span></tt> 表示列表的第二个元素，以此类推。
         * </p>
         * <p>
         * 你也可以使用负数下标，以 <tt class="docutils literal"><span class="pre">-1</span></tt> 表示列表的最后一个元素， <tt class="docutils literal"><span class="pre">-2</span></tt> 表示列表的倒数第二个元素，以此类推。
         * </p>
         * <p>
         * <strong>注意LRANGE命令和编程语言区间函数的区别</strong>
         * </p>
         * <p>
         * 假如你有一个包含一百个元素的列表，对该列表执行
         * <tt class="docutils literal"><span class="pre">LRANGE</span> <span class="pre">list</span> <span class="pre">0</span> <span class="pre">10</span></tt>
         * ，结果是一个包含11个元素的列表，这表明 <tt class="docutils literal"><span class="pre">stop</span></tt> 下标也在 <a class="reference internal" href="#lrange">LRANGE</a>
         * 命令的取值范围之内(闭区间)，这和某些语言的区间函数可能不一致，比如Ruby的 <tt class="docutils literal"><span class="pre">Range.new</span></tt> 、
         * <tt class="docutils literal"><span class="pre">Array#slice</span></tt> 和Python的 <tt class="docutils literal"><span class="pre">range()</span></tt> 函数。
         * </p>
         * <p>
         * <strong>超出范围的下标</strong>
         * </p>
         * <p>
         * 超出范围的下标值不会引起错误。
         * </p>
         * <p>
         * 如果 <tt class="docutils literal"><span class="pre">start</span></tt> 下标比列表的最大下标 <tt class="docutils literal"><span class="pre">end</span></tt> (
         * <tt class="docutils literal"><span class="pre">LLEN</span> <span class="pre">list</span></tt> 减去 <tt class="docutils literal"><span class="pre">1</span></tt> )还要大，或者
         * <tt class="docutils literal"><span class="pre">start</span> <span class="pre">&gt;</span> <span class="pre">stop</span></tt> ， <a class="reference internal"
         * href="#lrange">LRANGE</a> 返回一个空列表。
         * </p>
         * <p>
         * 如果 <tt class="docutils literal"><span class="pre">stop</span></tt> 下标比 <tt class="docutils literal"><span class="pre">end</span></tt> 下标还要大，Redis将
         * <tt class="docutils literal"><span class="pre">stop</span></tt> 的值设置为 <tt class="docutils literal"><span class="pre">end</span></tt> 。
         * </p>
         * 
         * @param start -
         * @param end -
         * @return -
         * @author 陈祥 2015-2-11
         */
        List<String> lrange(long start, long end);

        /**
         * 从指定位置开始获取后边的所有数据
         * 
         * @param start -
         * @return -
         * @author 陈祥 2019-8-10
         */
        List<String> lrange(long start);

        /**
         * 获取所有数据
         * 
         * @return -
         * @author 陈祥 2019-8-10
         */
        List<String> lrange();

        /**
         * <p>
         * <strong>LREM key count value</strong>
         * </p>
         * <p>
         * 根据参数 <tt class="docutils literal"><span class="pre">count</span></tt> 的值，移除列表中与参数 <tt class="docutils literal"><span class="pre">value</span></tt> 相等的元素。
         * </p>
         * <p>
         * <tt class="docutils literal"><span class="pre">count</span></tt> 的值可以是以下几种：
         * </p>
         * <ul class="simple">
         * <li><tt class="docutils literal"><span class="pre">count</span> <span class="pre">&gt;</span> <span class="pre">0</span></tt> : 从表头开始向表尾搜索，移除与
         * <tt class="docutils literal"><span class="pre">value</span></tt> 相等的元素，数量为 <tt class="docutils literal"><span class="pre">count</span></tt> 。</li>
         * <li><tt class="docutils literal"><span class="pre">count</span> <span class="pre">&lt;</span> <span class="pre">0</span></tt> : 从表尾开始向表头搜索，移除与
         * <tt class="docutils literal"><span class="pre">value</span></tt> 相等的元素，数量为 <tt class="docutils literal"><span class="pre">count</span></tt> 的绝对值。</li>
         * <li><tt class="docutils literal"><span class="pre">count</span> <span class="pre">=</span> <span class="pre">0</span></tt> : 移除表中所有与
         * <tt class="docutils literal"><span class="pre">value</span></tt> 相等的值。</li>
         * </ul>
         * <dl class="docutils">
         * <dt><strong>时间复杂度：</strong></dt>
         * <dd>O(N)， <tt class="docutils literal"><span class="pre">N</span></tt> 为列表的长度。</dd>
         * <dt><strong>返回值：</strong></dt>
         * <dd><div class="first last line-block"> <div class="line">被移除元素的数量。</div> <div class="line">因为不存在的 <tt class="docutils literal"><span class="pre">key</span></tt>
         * 被视作空表(empty list)，所以当 <tt class="docutils literal"><span class="pre">key</span></tt> 不存在时， <a class="reference internal" href="#lrem">LREM</a> 命令总是返回
         * <tt class="docutils literal"><span class="pre">0</span></tt> 。</div> </div></dd>
         * </dl>
         * 
         * @param count -
         * @param value -
         * @return -
         * @author 陈祥 2015-2-11
         */
        long lrem(long count, String value);
    }

    /**
     * 返回key类型为list的数量
     * 
     * @param key -
     * @return list中的数量
     * @author dengjinye 2015-5-25
     */
    long llen(String key);

    /**
     * 从key中从左边pop一个值
     * 
     * @param key -
     * @return -
     * @author dengjinye 2015-5-25
     */
    String lpop(String key);

    /**
     * 从左边push值到指定的key中
     * 
     * @param key -
     * @param val -
     * @return -
     * @author dengjinye 2015-5-25
     */
    long lpush(String key, String... val);

    /**
     * 从key中从右边pop一个值
     * 
     * @param key -
     * @return -
     * @author dengjinye 2015-5-25
     */
    String rpop(String key);

    /**
     * 从右边push值到指定的key中
     * 
     * @param key -
     * @param val -
     * @return -
     * @author dengjinye 2015-5-25
     */
    long rpush(String key, String... val);
}
