/*
 * Copyright 2024 Taskflow, Inc.
 * <p>
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * <p>
 * http://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package cn.feiliu.protogen.protobuf.compiler;

import java.io.BufferedReader;
import java.io.File;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.List;

public class ProtobufCompiler {
    private final String protoSourceRoot;
    private final String outputDirectory;
    private final String protocPath;
    private final String grpcPluginPath;

    public ProtobufCompiler(String protoSourceRoot, String outputDirectory, String protocPath, String grpcPluginPath) {
        this.protoSourceRoot = protoSourceRoot;
        this.outputDirectory = outputDirectory;
        this.protocPath = protocPath;
        this.grpcPluginPath = grpcPluginPath;
    }

    public void compile() throws Exception {
        // 确保输出目录存在
        new File(outputDirectory).mkdirs();

        // 构建编译命令
        List<String> command = new ArrayList<>();
        command.add(protocPath);

        // 添加 Google Protobuf 标准类型的包含路径
        String googleProtosPath = resolveGoogleProtos();
        command.add("--proto_path=" + googleProtosPath);

        // 添加项目的 proto 文件路径
        command.add("--proto_path=" + new File(protoSourceRoot).getAbsolutePath());
        command.add("--java_out=" + outputDirectory);

        if (grpcPluginPath != null) {
            command.add("--plugin=protoc-gen-grpc-java=" + grpcPluginPath);
            command.add("--grpc-java_out=" + outputDirectory);
        }

        // 添加所有.proto文件
        File protoDir = new File(protoSourceRoot);
        if (!protoDir.exists() || !protoDir.isDirectory()) {
            throw new RuntimeException("Proto source directory not found: " + protoSourceRoot);
        }
        List<File> protoFiles = findProtoFiles(protoDir);
        if (protoFiles.isEmpty()) {
            throw new RuntimeException("No .proto files found in " + protoSourceRoot);
        }

        // 打印命令和文件列表以便调试
        System.out.println("Proto files found:");
        for (File file : protoFiles) {
            System.out.println(" - " + file.getAbsolutePath());
        }

        for (File protoFile : protoFiles) {
            command.add(protoFile.getAbsolutePath());
        }

        // 打印完整命令
        System.out.println("Executing command: " + String.join(" ", command));

        // 执行命令
        ProcessBuilder processBuilder = new ProcessBuilder(command);
        processBuilder.redirectErrorStream(true);
        Process process = processBuilder.start();

        // 读取输出
        StringBuilder output = new StringBuilder();
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream()))) {
            String line;
            while ((line = reader.readLine()) != null) {
                output.append(line).append("\n");
                System.out.println(line); // 打印编译过程输出
            }
        }

        int exitCode = process.waitFor();
        if (exitCode != 0) {
            throw new RuntimeException("Protobuf compilation failed with exit code: " + exitCode + "\nOutput:\n"
                                       + output.toString());
        }
    }

    private String resolveGoogleProtos() throws Exception {
        // 从 Maven 仓库解析 google-protobuf
        String googleProtosPath = MavenProtocResolver.resolveGoogleProtos();
        if (googleProtosPath == null) {
            throw new RuntimeException("Failed to resolve Google Protobuf standard types");
        }
        return googleProtosPath;
    }

    private List<File> findProtoFiles(File dir) {
        List<File> protoFiles = new ArrayList<>();
        File[] files = dir.listFiles();
        if (files != null) {
            for (File file : files) {
                if (file.isDirectory()) {
                    protoFiles.addAll(findProtoFiles(file));
                } else if (file.getName().endsWith(".proto")) {
                    protoFiles.add(file);
                }
            }
        }
        return protoFiles;
    }
}