/*
 * Copyright 2024 Taskflow, Inc.
 * <p>
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * <p>
 * http://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package cn.feiliu.protogen.protobuf.compiler;

import cn.feiliu.protogen.config.CompilerConfig;
import cn.feiliu.protogen.config.ProtoGenConfiguration;

import java.io.File;

/**
 * @author SHOUSHEN.LUAN
 * @since 2024-12-17
 */
public class ProtobufCompilerUtil {
    private static final String DEFAULT_PROTOC_VERSION = "3.19.2";
    private static final String DEFAULT_GRPC_VERSION   = "1.42.1";

    public static void compileProto(ProtoGenConfiguration config) {
        try {
            // 检查配置
            validateConfig(config);
            CompilerConfig compilerConfig = config.getCompilerConfig();
            // 如果没有指定 protoc 路径，则从 Maven 仓库获取
            String protocPath = compilerConfig.getProtocPath();
            if (protocPath == null) {
                String protocVersion = compilerConfig.getProtocVersion() != null ? compilerConfig.getProtocVersion()
                    : DEFAULT_PROTOC_VERSION;
                protocPath = MavenProtocResolver.resolveProtoc(protocVersion);
            }

            // 如果需要生成 gRPC 代码且没有指定插件路径，则从 Maven 仓库获取
            String grpcPluginPath = null;
            if (compilerConfig.isGenerateGrpc()) {
                grpcPluginPath = compilerConfig.getGrpcJavaPluginPath();
                if (grpcPluginPath == null) {
                    String grpcVersion = compilerConfig.getGrpcVersion() != null ? compilerConfig.getGrpcVersion()
                        : DEFAULT_GRPC_VERSION;
                    grpcPluginPath = MavenProtocResolver.resolveGrpcJavaPlugin(grpcVersion);
                }
            }

            // 如果配置了清理输出目录
            if (compilerConfig.isClearOutputDirectory()) {
                File outputDir = config.getBaseConfig().getOutputDirectoryFile();
                if (outputDir.exists()) {
                    deleteDirectory(outputDir);
                }
                outputDir.mkdirs();
            }

            // 创建编译器实例
            ProtobufCompiler compiler = new ProtobufCompiler(config.getBaseConfig().getProtoSourceRootFile()
                .getAbsolutePath(), config.getBaseConfig().getOutputDirectoryFile().getAbsolutePath(), protocPath,
                grpcPluginPath);

            // 执行编译
            compiler.compile();

        } catch (Exception e) {
            throw new RuntimeException("Failed to compile proto files", e);
        }
    }

    private static void validateConfig(ProtoGenConfiguration config) {
        File file = config.getBaseConfig().getOutputDirectoryFile();
        if (!file.exists()) {
            throw new IllegalArgumentException("outputDirectory is required");
        }
        // 检查源目录是否存在
        File sourceDir = config.getBaseConfig().getProtoSourceRootFile();
        if (!sourceDir.exists() || !sourceDir.isDirectory()) {
            throw new IllegalArgumentException("Proto source directory does not exist: " + sourceDir.getAbsolutePath());
        }

        // 检查是否有 .proto 文件
        if (!hasProtoFiles(sourceDir)) {
            throw new IllegalArgumentException("No .proto files found in source directory: "
                                               + sourceDir.getAbsolutePath());
        }
    }

    private static boolean hasProtoFiles(File dir) {
        File[] files = dir.listFiles();
        if (files != null) {
            for (File file : files) {
                if (file.isDirectory()) {
                    if (hasProtoFiles(file)) {
                        return true;
                    }
                } else if (file.getName().endsWith(".proto")) {
                    return true;
                }
            }
        }
        return false;
    }

    private static void deleteDirectory(File directory) {
        File[] files = directory.listFiles();
        if (files != null) {
            for (File file : files) {
                if (file.isDirectory()) {
                    deleteDirectory(file);
                } else {
                    file.delete();
                }
            }
        }
        directory.delete();
    }
}