/**
 * Copyright (c) 2019,2020 honintech
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the “Software”), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 */
package cn.weforward.boot;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.support.PropertySourcesPlaceholderConfigurer;
import org.springframework.core.io.ClassPathResource;

import cn.weforward.common.util.StringUtil;

/**
 * 云配置
 * 
 * @author daibo
 *
 */
public class CloudPropertyPlaceholderConfigurer extends PropertySourcesPlaceholderConfigurer {
	/** 日志 */
	static final Logger _Logger = LoggerFactory.getLogger(CloudPropertyPlaceholderConfigurer.class);
	/** 是否禁用云配置 */
	boolean m_DisableCloud = false;
	/** 是否禁用云配置缓存 */
	boolean m_DisableCloudCache = false;
	/** 云配置缓存路径 */
	protected String m_CloudCachePath = null;

	/** 禁止云配置功能 */
	private final static String CLOUD_CONFIG_DISABLE = "cloud.disable";
	/** 禁止云配置缓存功能 */
	private final static String CLOUD_CONFIG_CACHE_DISABLE = "cloud.cache.disable";
	/** 云配置缓存路径 */
	private final static String CLOUD_CONFIG_CACHE_PATH = "cloud.cache.path";
	/** 开发配置文件key */
	private final static String DEV_KEY = "weforward-dev.properties";
	/** 测试配置文件key */
	private final static String TEST_PROP_KEY = "weforward-test.properties";
	/** 配置文件key */
	private final static String PROP_KEY = "weforward.properties";

	private CloudLoader m_CloudLoader;

	public CloudPropertyPlaceholderConfigurer() {
		m_CloudLoader = new CloudLoader();
		setDisableCloud("true".equalsIgnoreCase(System.getProperty(CLOUD_CONFIG_DISABLE)));
		setDisableCloudCache("true".equalsIgnoreCase(System.getProperty(CLOUD_CONFIG_CACHE_DISABLE)));
		String path = System.getProperty(CLOUD_CONFIG_CACHE_PATH);
		if (StringUtil.isEmpty(path)) {
			path = System.getProperty("user.dir") + File.separator + "config" + File.separator + File.separator;
		}
		m_CloudCachePath = path;
	}

	/**
	 * 是否禁止云配置功能
	 * 
	 * @param disable true=禁止，false=不禁止
	 */
	public void setDisableCloud(boolean disable) {
		m_DisableCloud = disable;
	}

	/**
	 * 是否禁止云配置缓存功能
	 * 
	 * @param disable true=禁止，false=不禁止
	 */
	public void setDisableCloudCache(boolean disable) {
		m_DisableCloudCache = disable;
	}

	@Override
	protected Properties mergeProperties() throws IOException {
		// 优先级 setLocaion > weforward-dev.properties > weforward-test.properties >
		// weforward.properties > Service
		// properties
		Properties result = super.mergeProperties();
		try {
			result = load(result, DEV_KEY);
		} catch (Throwable e) {
			throw new IllegalArgumentException("获取配置失败:" + e.getMessage(), e);
		}
		try {
			result = load(result, TEST_PROP_KEY);
		} catch (Throwable e) {
			throw new IllegalArgumentException("获取配置失败:" + e.getMessage(), e);
		}
		try {
			result = load(result, PROP_KEY);
		} catch (Throwable e) {
			throw new IllegalArgumentException("获取配置失败:" + e.getMessage(), e);
		}
		if (!m_DisableCloud) {
			Properties cloudresult = null;
			try {
				cloudresult = m_CloudLoader.load(result);
				if (!m_DisableCloudCache) {
					updateCloudCache(result);
				}
			} catch (RuntimeException e) {
				if (!m_DisableCloudCache) {
					_Logger.warn("加载云配置失败，使用本地缓存", e);
					cloudresult = loadCloudCache();
				}
				if (null == cloudresult) {
					_Logger.warn("未找到本地缓存");
					throw e;
				}
			}
			for (Map.Entry<Object, Object> e : cloudresult.entrySet()) {
				result.putIfAbsent(e.getKey(), e.getValue());
			}
		}
		return result;
	}

	private File getCloudCacheFile() {
		File path = new File(m_CloudCachePath);
		if (!path.exists()) {
			path.mkdirs();
		}
		return new File(path, "cloud.properties");
	}

	private Properties loadCloudCache() throws IOException {
		File file = getCloudCacheFile();
		if (!file.exists()) {
			return null;
		}
		try (BufferedReader read = new BufferedReader(new FileReader(file))) {
			Properties result = new Properties();
			String line;
			while (null != (line = read.readLine())) {
				int index = line.indexOf('=');
				String key = line.substring(0, index);
				String value = line.substring(index + 1);
				result.put(key, value);
			}
			return result;
		} catch (IOException e) {
			throw e;
		}

	}

	private void updateCloudCache(Properties result) throws IOException {
		File file = getCloudCacheFile();
		try (BufferedWriter out = new BufferedWriter(new FileWriter(file))) {
			for (Map.Entry<Object, Object> e : result.entrySet()) {
				out.write(StringUtil.toString(e.getKey()) + "=" + StringUtil.toString(e.getValue()));
				out.newLine();
			}
		} catch (IOException e) {
			throw e;
		}

	}

	@Override
	protected void loadProperties(Properties props) throws IOException {
		try {
			super.loadProperties(props);
		} catch (FileNotFoundException e) {
			_Logger.warn("忽略找不到文件异常:" + e.getMessage());
		}
	}

//	@Override
//	protected void processProperties(ConfigurableListableBeanFactory beanFactoryToProcess, Properties props)
//			throws BeansException {
//		super.processProperties(beanFactoryToProcess, props);
//		String dataPath = props.getProperty("data.path");
//		if (!StringUtil.isEmpty(dataPath)) {
//			File file = new File(FileUtil.getAbsolutePath(dataPath, null));
//			if (!file.exists()) {
//				if (file.mkdir()) {
//					_Logger.info("自动创建数据目录:" + file.getAbsolutePath());
//				}
//			}
//		}
//	}

	/* 加载资源 */
	private static Properties load(Properties result, String propkey) throws IOException {
		ClassPathResource res = new ClassPathResource(propkey);
		if (res.exists()) {
			_Logger.info("load " + propkey);
			Properties prop = new Properties();
			prop.load(res.getInputStream());
			for (Entry<Object, Object> e : prop.entrySet()) {
				String key = StringUtil.toString(e.getKey());
				if (null != result.get(key) || null != System.getProperty(key)) {
					continue;
				}
				String value = StringUtil.toString(e.getValue());
				result.put(key, value);
			}
		}
		return result;

	}

}
