package cn.wjee.commons.collection;

import cn.wjee.commons.functional.KeyValueConsumer;
import cn.wjee.commons.lang.RandomUtils;
import cn.wjee.commons.lang.ReflectUtils;
import cn.wjee.commons.lang.StringUtils;

import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

/**
 * @author wjee
 * @version $Id: CommonCollectionUtils.java, v 0.1 2015年9月10日 上午11:27:53 wjee Exp $
 * Common Collection Utils
 */
public class CollectionUtils {
    private CollectionUtils() {
    }

    /**
     * 集合是否为空
     *
     * @param coll 集合
     * @return boolean
     */
    public static boolean isEmpty(Collection<?> coll) {
        return coll == null || coll.isEmpty();
    }

    /**
     * 集合是否非空
     *
     * @param coll 集合
     * @return boolean
     */
    public static boolean isNotEmpty(Collection<?> coll) {
        return !isEmpty(coll);
    }

    /**
     * 将字符串按照分隔符切割成数组
     *
     * @param str               内容
     * @param trimTokens        结果是否去空格
     * @param delimiters        分隔符
     * @param ignoreEmptyTokens 忽略空结果
     * @return String[]
     */
    public static String[] tokenizeToArray(String str, String delimiters, boolean trimTokens, boolean ignoreEmptyTokens) {
        if (str == null) {
            return new String[]{};
        }
        StringTokenizer st = new StringTokenizer(str, delimiters);
        List<String> tokens = new ArrayList<>();
        while (st.hasMoreTokens()) {
            String token = st.nextToken();
            if (trimTokens) {
                token = token.trim();
            }
            if (!ignoreEmptyTokens || !token.isEmpty()) {
                tokens.add(token);
            }
        }
        return tokens.toArray(new String[0]);
    }

    /**
     * 字符串根据分隔符转数组
     *
     * @param content    内容
     * @param delimiters 分割符
     * @return String[]
     */
    public static String[] tokenizeToArray(String content, String delimiters) {
        return tokenizeToArray(content, delimiters, true, true);
    }

    /**
     * 字符串根据默认分隔符转列表
     *
     * @param content    内容
     * @param delimiters 分隔符
     * @return List
     */
    public static List<String> tokenizeToList(String content, String delimiters) {
        return Arrays.asList(tokenizeToArray(content, delimiters));
    }

    /**
     * String convert to Set by delimiters
     *
     * @param content    内容
     * @param delimiters 分割符
     * @return Set
     */
    public static Set<String> tokenizeToSet(String content, String delimiters) {
        return new HashSet<>(tokenizeToList(content, delimiters));
    }

    /**
     * Convert List to uppercase List
     *
     * @param list 源类表
     * @return List
     */
    public static List<String> upperList(List<String> list) {
        return list.parallelStream().filter(StringUtils::isNotBlank).map(StringUtils::upperCase).collect(Collectors.toList());
    }

    /**
     * 将列表分割为嵌套列表
     *
     * @param dataList 数据列表
     * @param size     子列表大小
     * @param <T>      类型
     * @return List
     */
    public static <T> List<List<T>> splitList(List<T> dataList, int size) {
        if (CollectionUtils.isEmpty(dataList)) {
            return new ArrayList<>();
        }
        List<List<T>> resultList = new ArrayList<>();
        if (size < 1) {
            resultList.add(dataList);
            return resultList;
        }
        int totalCount = dataList.size();
        int subListCount = totalCount % size == 0 ? (totalCount / size) : (totalCount / size + 1);
        for (int i = 0; i < subListCount; i++) {
            int start = i * size;
            int end = (i + 1) * size;
            resultList.add(dataList.subList(start, Math.min(end, totalCount)));
        }
        return resultList;
    }

    /**
     * 分割去重转数组
     *
     * @param values 值
     * @return String[]
     */
    public static String[] tokenizeToUniqueArray(String values) {
        return CollectionUtils.tokenizeToSet(values, ",").toArray(new String[0]);
    }

    /**
     * 数组随机取N个不重复下标元素
     *
     * @param list list
     * @param size 随机挑选元素个数
     * @param <T>  泛型
     * @return List
     */
    public static <T> List<T> randomSubList(List<T> list, int size) {
        List<T> resultList = new ArrayList<>();
        if (CollectionUtils.isEmpty(list)) {
            return resultList;
        }
        if (list.size() <= size) {
            resultList.addAll(list);
            return resultList;
        }
        Set<Integer> indexSet = new HashSet<>();
        while (indexSet.size() < size) {
            int index = RandomUtils.nextInt(0, list.size());
            indexSet.add(index);
        }
        indexSet.forEach(item -> resultList.add(list.get(item)));
        return resultList;
    }

    /**
     * 是否存在任一个集合为空
     *
     * @param collections 集合数组
     * @return boolean
     */
    public static boolean isAnyEmpty(Collection<?>... collections) {
        if (collections == null || collections.length == 0) {
            return true;
        }
        return Arrays.stream(collections).anyMatch(CollectionUtils::isEmpty);
    }

    /**
     * 待检测集合中某个元素是否出现在源集合中
     *
     * @param srcList   源集合
     * @param checkList 待检测集合
     * @param <T>       泛型
     * @return boolean
     */
    public static <T> boolean containsAny(Collection<T> srcList, Collection<T> checkList) {
        if (isAnyEmpty(srcList, checkList)) {
            return false;
        }
        return checkList.stream().anyMatch(srcList::contains);
    }

    /**
     * 是否包含，忽略大小写
     *
     * @param srcList 源集合
     * @param value   检测值
     * @return boolean
     */
    public static boolean containsIgnoreCase(Collection<String> srcList, String value) {
        return srcList.stream().anyMatch(item -> StringUtils.isNotBlank(value) && StringUtils.equalsIgnoreCase(item, value));
    }

    /**
     * 打散抽取工具方法(默认不丢弃抽取结果)
     *
     * @param dataList  数据列表
     * @param distKey   打散字段
     * @param distTimes 抽取轮数
     * @param distCount 单次抽取文档数
     * @param <T>       泛型
     * @return List
     */
    public static <T> List<T> distinct(List<T> dataList, String distKey, Integer distTimes, Integer distCount) {
        return distinct(dataList, distKey, distTimes, distCount, false);
    }

    /**
     * 打散抽取工具方法.
     * <p>
     * 举例：[{doc1,张三}, {doc2,张三}, {doc3, 张三}, {doc4,李四}, {doc5,王五}, {doc6,王五}, {doc7,赵6}]
     * <p>
     * 根据名字抽取1轮1次丢弃抽取后数据 - doc1,doc4,doc5,doc7
     * 根据名字抽取2轮1次丢弃抽取后数据 - doc1,doc4,doc5,doc7,doc2,doc6
     * 根据名字抽取2轮1次不丢弃抽取后数据 - doc1,doc4,doc5,doc7,doc2,doc6,  doc3
     *
     * @param dataList  数据列表
     * @param distKey   打散字段
     * @param distTimes 抽取轮数
     * @param distCount 单次抽取文档数
     * @param reversed  是否丢弃抽取后数据
     * @param <T>       泛型
     * @return List
     */
    public static <T> List<T> distinct(List<T> dataList, String distKey, Integer distTimes, Integer distCount, boolean reversed) {
        final List<Integer> finalIndexList = new ArrayList<>();
        for (int times = 0; times < distTimes; times++) {
            int sameCount = 0;
            Object lastDistKeyValue = null;
            for (int index = 0; index < dataList.size(); index++) {
                // 打散列值
                final Object tempDistFieldValue = ReflectUtils.getProperty(dataList.get(index), distKey);
                // 首位默认添加
                if (times == 0 && index == 0) {
                    finalIndexList.add(0);
                    sameCount = 1;
                    lastDistKeyValue = tempDistFieldValue;
                    continue;
                }
                // 已进标记队列的忽略
                if (finalIndexList.contains(index)) {
                    continue;
                }
                // (上次打散字段值为Null || 本值与上次不一致),添加本次为初始
                if (lastDistKeyValue == null || !lastDistKeyValue.equals(tempDistFieldValue)) {
                    sameCount = 1;
                    lastDistKeyValue = tempDistFieldValue;
                    finalIndexList.add(index);
                    continue;
                }
                // 与上次值一致的情况, 抽取额度未满添加, 否则继续下一个
                if (sameCount < distCount) {
                    sameCount += 1;
                    finalIndexList.add(index);
                }
            }
        }
        final List<T> finalList = new ArrayList<>();
        finalIndexList.forEach(integer -> finalList.add(dataList.get(integer)));
        // 是否丢弃抽取后的数据
        if (!reversed) {
            AtomicInteger atomicInteger = new AtomicInteger(0);
            List<Integer> allIdList = new ArrayList<>();
            while (allIdList.size() < dataList.size()) {
                allIdList.add(atomicInteger.getAndAdd(1));
            }
            allIdList.stream().filter(i -> !finalIndexList.contains(i)).forEach(t -> finalList.add(dataList.get(t)));
        }
        return finalList;
    }

    /**
     * 集合嵌套循环
     *
     * @param collectionA 集合A
     * @param collectionB 集合B
     * @param consumer    回调
     * @param <E>         泛型A
     * @param <F>         泛型B
     */
    public <E, F> void join(Collection<E> collectionA, Collection<F> collectionB, KeyValueConsumer<E, F> consumer) {
        if (isNotEmpty(collectionA) && isNotEmpty(collectionB)) {
            for (E a : collectionA) {
                for (F b : collectionB) {
                    consumer.accept(a, b);
                }
            }
        } else if (isNotEmpty(collectionA)) {
            for (E a : collectionA) {
                consumer.accept(a, null);
            }
        } else if (isNotEmpty(collectionB)) {
            for (F b : collectionB) {
                consumer.accept(null, b);
            }
        }
    }
}
