package cn.wjee.commons.collection;

import cn.wjee.commons.crypto.EncodeUtils;
import cn.wjee.commons.domain.Paging;
import cn.wjee.commons.exception.BizException;
import cn.wjee.commons.lang.StringUtils;
import cn.wjee.commons.lang.html.HtmlUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 参数获取工具类
 *
 * @author listening
 */
@SuppressWarnings("rawtypes")
public class MapUtils {
    private MapUtils() {
    }

    /**
     * 构造Map的Builder工具
     *
     * @param key   键
     * @param value 值
     * @param <K>   键泛型
     * @param <V>   值泛型
     * @return MapBuilder
     */
    public static <K, V> MapBuilder<K, V> of(K key, V value) {
        return MapBuilder.of(key, value);
    }

    /**
     * 创建RequestMap
     *
     * @return RequestMap
     */
    public static RequestMap newRequestMap() {
        return new RequestMap();
    }

    /**
     * 创建多值Map
     *
     * @param <K> K
     * @param <V> V
     * @return MultiMap
     */
    public static <K, V> MultiMap<K, V> newMultiMap() {
        return new MultiMap<>();
    }

    /**
     * Map是否为空
     *
     * @param paramMap 参数MAP
     * @return boolean
     */
    public static boolean isEmpty(Map paramMap) {
        return paramMap == null || paramMap.isEmpty();
    }

    /**
     * 判断Map是否非空
     *
     * @param paramMap 参数Map
     * @return boolean
     */
    public static boolean isNotEmpty(Map paramMap) {
        return !isEmpty(paramMap);
    }

    /**
     * 参数按照Ascii字母排序
     *
     * @param paramMap    参数Map
     * @param isAsc       是否升序
     * @param isUrlEncode 是否URL转码
     * @return String
     */
    public static String paramAsciiOrder(Map<String, String> paramMap, boolean isAsc, boolean isUrlEncode) {
        List<Map.Entry<String, String>> entryList = new ArrayList<>(paramMap.entrySet());
        entryList.sort(Map.Entry.comparingByKey());
        if (!isAsc) {
            Collections.reverse(entryList);
        }
        StringBuilder builder = new StringBuilder();
        for (Map.Entry<String, String> entry : entryList) {
            String value = isUrlEncode ? EncodeUtils.urlEncode(entry.getValue()) : entry.getValue();
            builder.append("&").append(entry.getKey()).append("=").append(value);
        }
        builder.deleteCharAt(0);
        return builder.toString();
    }

    /**
     * 从MAP取值String
     *
     * @param map      参数MAP
     * @param key      参数编码
     * @param defaults 默认值
     * @param isSafeH5 是否启用H5转义
     * @return String
     */
    public static String getValue(Map map, Object key, Object defaults, boolean isSafeH5) {
        if (map == null || key == null) {
            throw new BizException("参数Map和查询Key不能为空");
        }
        Object obj = map.get(key);
        if (obj == null) {
            obj = defaults;
        }
        String value = StringUtils.getValue(String.valueOf(obj));
        return isSafeH5 ? HtmlUtils.htmlEscape(value) : value;
    }

    /**
     * 从MAP取值
     *
     * @param map 参数MAP
     * @param key 参数编码
     * @return String
     */
    public static String getValue(Map map, Object key) {
        return getValue(map, key, null, true);
    }

    /**
     * 从MAP取值
     *
     * @param map      参数MAP
     * @param key      参数编码
     * @param defaults 默认值
     * @return String
     */
    public static String getValue(Map map, String key, Object defaults) {
        return getValue(map, key, defaults, true);
    }

    /**
     * Get Integer Value From ParamMap
     *
     * @param map 参数MAP
     * @param key 参数编码
     * @return Integer
     */
    public static Integer getIntValue(Map map, String key) {
        return getIntValue(map, key, null);
    }

    /**
     * Get Integer Value From ParamMap
     *
     * @param map      参数MAP
     * @param key      参数编码
     * @param defaults 默认值
     * @return Integer
     */
    public static Integer getIntValue(Map map, String key, Integer defaults) {
        return StringUtils.getIntValue(getValue(map, key, defaults, false), defaults);
    }

    /**
     * Get Long Value From ParamMap
     *
     * @param map 参数MAP
     * @param key 参数编码
     * @return Long
     */
    public static Long getLongValue(Map map, String key) {
        return StringUtils.getLongValue(getValue(map, key, false));
    }

    /**
     * Get Long Value From ParamMap
     *
     * @param map      参数MAP
     * @param key      参数编码
     * @param defaults 默认值
     * @return Long
     */
    public static Long getLongValue(Map map, String key, Long defaults) {
        return StringUtils.getLongValue(getValue(map, key, defaults, false), defaults);
    }

    /**
     * 获取分页属性
     *
     * @param map          参数Map
     * @param pageField    参数名
     * @param limitField   值
     * @param defaultPage  默认页码
     * @param defaultLimit 默认页面大小
     * @return Paging
     */
    public static Paging getPaging(Map map, String pageField, String limitField, int defaultPage, int defaultLimit) {
        int pageNo = getIntValue(map, pageField, defaultPage);
        int pageSize = getIntValue(map, limitField, defaultLimit);
        if (pageNo <= 0) {
            pageNo = defaultPage;
        }
        if (pageSize <= 0) {
            pageSize = defaultLimit;
        }
        return new Paging(pageNo, pageSize);
    }

    /**
     * 获取分页信息
     *
     * @param paramMap       参数Map
     * @param pageNoField    参数名
     * @param pageLimitField 值
     * @return Paging
     */
    public static Paging getPaging(Map paramMap, String pageNoField, String pageLimitField) {
        return getPaging(paramMap, pageNoField, pageLimitField, 1, 20);
    }

    /**
     * Map反转(K,V) to (V,K), 重复取第一个
     *
     * @param src 源Map
     * @param <K> K
     * @param <V> V
     * @return Map
     */
    public static <K, V> Map<V, K> reverseHashMap(Map<K, V> src) {
        // 转成Map<V, List<Entry>>
        Map<V, Set<K>> vkGroupMap = Optional.ofNullable(src).orElse(new HashMap<>(0))
            .entrySet().stream()
            .collect(Collectors.groupingBy(Map.Entry::getValue,
                Collectors.mapping(Map.Entry::getKey, Collectors.toSet())));

        Map<V, K> result = new HashMap<>(0);
        for (Map.Entry<V, Set<K>> tempEntry : vkGroupMap.entrySet()) {
            if (tempEntry.getKey() == null) {
                continue;
            }

            Set<K> valueSet = tempEntry.getValue();
            if (valueSet.stream().noneMatch(Objects::nonNull)) {
                continue;
            }

            K k = tempEntry.getValue()
                .stream()
                .filter(Objects::nonNull)
                .findFirst().orElse(null);
            if (k == null) {
                continue;
            }

            result.put(tempEntry.getKey(), k);
        }
        return result;
    }
}
