package cn.wjee.commons.collection;

import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.NoArgsConstructor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

/**
 * 步骤跟踪信息
 *
 * @author listening
 */
public class StepWatch {
    /**
     * 日志
     */
    private static final Logger log = LoggerFactory.getLogger(StepWatch.class);

    private final boolean traceOnEvent;
    /**
     * 名称
     */
    private final String name;
    /**
     * 开始时间
     */
    private final Long startTime;
    /**
     * 上一步
     */
    private StepDebug lastStep;
    /**
     * 阶段列表
     */
    private final List<StepDebug> stepList;

    public StepWatch() {
        this(UUID.randomUUID().toString());
    }

    public StepWatch(String name) {
        this(name, false);
    }

    public StepWatch(String name, boolean traceOnEvent) {
        this.name = name;
        this.startTime = System.currentTimeMillis();
        this.stepList = new ArrayList<>();
        this.traceOnEvent = traceOnEvent;
    }

    public static StepWatch startWatch(String name) {
        return new StepWatch(name);
    }

    public static StepWatch startWatch(String name, boolean traceOnEvent) {
        return new StepWatch(name, traceOnEvent);
    }

    @Data
    private static class StepDebug {
        /**
         * 阶段名
         */
        private String name;
        /**
         * 执行时间点
         */
        private Long timestamp;
        /**
         * 耗时
         */
        private Long costTime;

        @Override
        public String toString() {
            return getName() + "--> " + getCostTime() + "ms";
        }
    }

    public void watch(String stepName) {
        try {
            StepDebug stepDebug = new StepDebug();
            stepDebug.setName(stepName);
            stepDebug.setTimestamp(System.currentTimeMillis());

            if (this.lastStep != null) {
                stepDebug.setCostTime(stepDebug.getTimestamp() - lastStep.getTimestamp());
            } else {
                stepDebug.setCostTime(stepDebug.getTimestamp() - startTime);
            }
            this.stepList.add(stepDebug);
            if (traceOnEvent) {
                log.info("[{}]-->{}", name, stepDebug);
            }
            this.lastStep = stepDebug;
        } catch (Exception e) {
            log.debug("watch fail", e);
        }
    }

    public String logs() {
        try {
            if (stepList == null || stepList.isEmpty()) {
                return "";
            }
            long maxTimeStamp = stepList.stream().mapToLong(StepDebug::getTimestamp).max().orElse(0);
            final long totalCost = maxTimeStamp - startTime;
            StringBuilder builder = new StringBuilder("\n");
            builder.append("---------------------------------------------").append("\n");
            builder.append("【").append(name).append("】-->耗时:").append(totalCost).append("ms").append("\n");
            builder.append("---------------------------------------------").append("\n");


            for (int i = 0; i < stepList.size(); i++) {
                StepDebug stepDebug = stepList.get(i);
                builder.append(i).append(". ").append(name)
                    .append("-->").append(stepDebug.toString()).append("\n");
            }
            return builder.toString();
        } catch (Exception ignored) {
            return "";
        }
    }

    public void trace() {
        String logs = logs();
        log.info(logs);
    }

    /**
     * 跟踪信息
     *
     * @author listening
     */
    @Data
    @AllArgsConstructor
    @NoArgsConstructor
    public static class StepTraceInfo {
        private String traceId;

        private Long costTime;

        private boolean success;

        public static StepTraceInfo of(String traceId, Long costTime, boolean success) {
            return new StepTraceInfo(traceId, costTime, success);
        }
    }
}
