package cn.wjee.commons.collection;

import cn.wjee.commons.domain.TreeNode;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 树工具
 *
 * @author listening
 */
public class TreeUtils {

    private TreeUtils() {

    }

    /**
     * 列表节点转树结构
     *
     * @param <K>    ID
     * @param list   节点列表
     * @param rootId 根节点
     * @return List
     */
    public static <K> List<TreeNode<K>> toTree(List<? extends TreeNode<K>> list, K rootId) {
        List<TreeNode<K>> rootNodeList = list.stream()
            .filter(item -> item.getParentId().equals(rootId))
            .collect(Collectors.toList());
        if (CollectionUtils.isEmpty(rootNodeList)) {
            return Collections.emptyList();
        }
        for (TreeNode<K> node : rootNodeList) {
            List<TreeNode<K>> treeNodes = toTree(list, node.getId());
            node.setChildren(treeNodes);
        }
        return rootNodeList;
    }

    /**
     * 数转换
     *
     * @param src       源数据
     * @param converter 转换器
     * @param <K>       泛型
     * @param <K1>      泛型
     * @return List
     */
    public static <K, K1> List<TreeNode<K1>> convert(List<? extends TreeNode<K>> src, Function<TreeNode<K>, TreeNode<K1>> converter) {
        if (src == null || src.isEmpty()) {
            return new ArrayList<>();
        }
        List<TreeNode<K1>> result = new ArrayList<>();
        for (TreeNode<K> node : src) {
            TreeNode<K1> apply = converter.apply(node);
            if (apply != null) {
                apply.setChildren(convert(node.getChildren(), converter));
                result.add(apply);
            }
        }
        return result;
    }

    /**
     * 树扁平化
     *
     * @param list 节点列表
     * @param <K>  泛型
     * @return List
     */
    public static <K> List<TreeNode<K>> flat(List<? extends TreeNode<K>> list) {
        if (CollectionUtils.isEmpty(list)) {
            return Collections.emptyList();
        }
        List<TreeNode<K>> result = new ArrayList<>();
        for (TreeNode<K> node : list) {
            List<? extends TreeNode<K>> children = node.getChildren();
            result.addAll(flat(children));
            node.setChildren(null);
            result.add(node);
        }
        return result;
    }
}
