package cn.wjee.commons.crypto;

import cn.wjee.commons.exception.Asserts;
import cn.wjee.commons.exception.BizException;
import cn.wjee.commons.lang.StringUtils;

import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;

/**
 * 编码工具类 实现aes加密、解密
 *
 * @author listening
 */
public class AES {
    /**
     * AES Cipher
     */
    private final Cipher cipher;

    private final AesType aesMode;

    private final String aesKey;

    private final String aesIv;

    public enum AesType {
        /**
         * CBC加密方式
         */
        CBC,
        /**
         * ECB加密方式
         */
        ECB
    }

    private AES(AesType mode, String key, String iv) {
        try {
            this.aesMode = mode;
            this.aesIv = iv;
            this.aesKey = key;
            cipher = Cipher.getInstance("AES/" + mode.name() + "/PKCS5Padding");
        } catch (NoSuchAlgorithmException | NoSuchPaddingException e) {
            throw new BizException("AES Cipher Construct fail", e);
        }
    }

    public static AES newCipherEcb(String key) {
        Asserts.isTrue(StringUtils.isNotBlank(key), "AES ECB配置错误");
        return new AES(AesType.ECB, key, null);
    }

    public static AES newCipherCbc(String key, String iv) {
        Asserts.isTrue(!StringUtils.isAnyBlank(key, iv), "AES CBC配置错误");
        return new AES(AesType.CBC, key, iv);
    }

    private void updateCipher(Integer cipherMode) {
        try {
            SecretKeySpec secretKeySpec = new SecretKeySpec(EncodeUtils.getBytes(aesKey), "AES");
            if (aesMode == AesType.ECB) {
                cipher.init(cipherMode, secretKeySpec);
            } else {
                Asserts.isTrue(StringUtils.isNotBlank(aesIv), "Aes CBC 模式下偏移量(iv)不能为空");
                IvParameterSpec ivParameterSpec = new IvParameterSpec(EncodeUtils.getBytes(aesIv));
                cipher.init(cipherMode, secretKeySpec, ivParameterSpec);
            }
        } catch (InvalidKeyException | InvalidAlgorithmParameterException e) {
            throw new BizException("AES update cipher fail", e);
        }
    }

    /**
     * 使用AES进行加密
     *
     * @param content 内容
     * @return String 加密后转Base64
     */
    public String encrypt(String content) {
        try {
            updateCipher(Cipher.ENCRYPT_MODE);
            byte[] encryptBytes = cipher.doFinal(EncodeUtils.getBytes(content));
            return EncodeUtils.getString(EncodeUtils.encodeBase64(encryptBytes));
        } catch (IllegalBlockSizeException | BadPaddingException e) {
            throw new BizException("encrypt fail", e);
        }
    }

    /**
     * 使用AES进行解密
     *
     * @param content 内容
     * @return String   解密后的内容
     */
    public String decrypt(String content) {
        try {
            updateCipher(Cipher.DECRYPT_MODE);
            byte[] decryptBytes = cipher.doFinal(EncodeUtils.decodeBase64(EncodeUtils.getBytes(content)));
            return EncodeUtils.getString(decryptBytes);
        } catch (IllegalBlockSizeException | BadPaddingException e) {
            throw new BizException("decrypt fail", e);
        }
    }
}
