package cn.wjee.commons.crypto;

import lombok.AccessLevel;
import lombok.NoArgsConstructor;

import java.io.ByteArrayOutputStream;

/**
 * 封装各种格式的编码解码工具类.
 * <p>
 * 1.Commons-Codec的 hex/base64 编码 2.自制的base62 编码 3.Commons-Lang的xml/html escape
 * 4.JDK提供的URLEncoder
 *
 * @author listening
 */
@NoArgsConstructor(access = AccessLevel.PRIVATE)
public class Base62 {
    /**
     * 编码表
     */
    private static final char[] STANDARD_ENCODE_TABLE = {
        'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M',
        'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z',
        'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm',
        'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z',
        '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '+', '/'
    };
    /**
     * 解码表
     */
    private static final byte[] DECODE_TABLE = {
        -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
        -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
        -1, -1, -1, -1, -1, -1, -1, -1, -1, 62, -1, 62, -1, 63, 52, 53, 54,
        55, 56, 57, 58, 59, 60, 61, -1, -1, -1, -1, -1, -1, -1, 0, 1, 2, 3, 4,
        5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23,
        24, 25, -1, -1, -1, -1, 63, -1, 26, 27, 28, 29, 30, 31, 32, 33, 34,
        35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51
    };

    /**
     * Base62编码
     *
     * @param content 密文
     * @return String
     */
    public static String encodeBase62(String content) {
        byte[] data = EncodeUtils.getBytes(content);
        StringBuilder sb = new StringBuilder(data.length * 2);
        int pos = 0;
        int val = 0;
        for (byte datum : data) {
            val = (val << 8) | (datum & 0xFF);
            pos += 8;
            while (pos > 5) {
                char c = STANDARD_ENCODE_TABLE[val >> (pos -= 6)];
                sb.append(c == 'i' ? "ia" : (c == '+' ? "ib" : (c == '/' ? "ic" : c + "")));
                val &= ((1 << pos) - 1);
            }
        }
        if (pos > 0) {
            char c = STANDARD_ENCODE_TABLE[val << (6 - pos)];
            sb.append(c == 'i' ? "ia" : (c == '+' ? "ib" : (c == '/' ? "ic" : c + "")));
        }
        return sb.toString();
    }

    /**
     * Base62解码
     *
     * @param content 密文
     * @return String
     */
    public static String decodeBase62(String content) {
        char[] data = content.toCharArray();
        ByteArrayOutputStream os = new ByteArrayOutputStream(data.length);
        int pos = 0;
        int val = 0;
        for (int i = 0; i < data.length; i++) {
            char c = data[i];
            if (c == 'i') {
                c = data[++i];
                c = c == 'a' ? 'i' : (c == 'b' ? '+' : (c == 'c' ? '/' : data[--i]));
            }
            val = (val << 6) | DECODE_TABLE[c];
            pos += 6;
            while (pos > 7) {
                os.write(val >> (pos -= 8));
                val &= ((1 << pos) - 1);
            }
        }
        return EncodeUtils.getString(os.toByteArray());
    }
}
