package cn.wjee.commons.crypto;

import cn.wjee.commons.exception.BizException;
import cn.wjee.commons.lang.StringUtils;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Base64;

/**
 * 封装各种格式的编码解码工具类.
 *
 * @author listening
 */
public class EncodeUtils {
    private EncodeUtils() {
    }

    /**
     * 默认编码
     */
    public static final String DEFAULT_CHARSET = StandardCharsets.UTF_8.name();

    /**
     * Hex编码.
     *
     * @param input 字节数组
     * @return String
     */
    public static String encodeHex(String input) {
        return encodeHex(getBytes(input));
    }

    /**
     * Hex编码.
     *
     * @param input 字节数组
     * @return String
     */
    public static String encodeHex(byte[] input) {
        StringBuilder result = new StringBuilder();
        for (byte datum : input) {
            result.append(String.format("%02X", datum));
        }
        return result.toString();
    }

    /**
     * Hex解码.
     *
     * @param input 待解密字符串
     * @return byte[]
     */
    public static String decodeHex(String input) {
        int l = input.length() / 2;
        byte[] bytes = new byte[l];
        for (int i = 0; i < l; i++) {
            bytes[i] = Integer.valueOf(input.substring(i * 2, i * 2 + 2), 16).byteValue();
        }
        return getString(bytes);
    }

    /**
     * Base64编码.
     *
     * @param bytes 待加密字符串字节数组
     * @return String
     */
    public static byte[] encodeBase64(byte[] bytes) {
        return Base64.getEncoder().encode(bytes);
    }

    /**
     * Base64解码.
     *
     * @param bytes 字符串字节数组
     * @return byte[]
     */
    public static byte[] decodeBase64(byte[] bytes) {
        return Base64.getDecoder().decode(bytes);
    }

    /**
     * Base64编码.
     *
     * @param input 待加密字符串
     * @return String
     */
    public static String encodeBase64(String input) {
        return getString(encodeBase64(getBytes(input)));
    }

    /**
     * Base64解码.
     *
     * @param input 字符串
     * @return byte[]
     */
    public static String decodeBase64(String input) {
        return getString(decodeBase64(getBytes(input)));
    }

    /**
     * Base64编码, URL安全(将Base64中的URL非法字符'+'和'/'转为'-'和'_', 见RFC3548).
     *
     * @param data 明文
     * @return String
     */
    public static String encodeUrlSafeBase64(String data) {
        return getString(Base64.getUrlEncoder().encode(getBytes(data)));
    }

    /**
     * URL 编码, Encode默认为UTF-8.
     *
     * @param part 待处理字符串
     * @return String
     */
    public static String urlEncode(String part) {
        try {
            return URLEncoder.encode(part, DEFAULT_CHARSET);
        } catch (UnsupportedEncodingException e) {
            throw new BizException("urlEncode fail", e);
        }
    }

    /**
     * URL 解码, Encode默认为UTF-8.
     *
     * @param part 待处理字符串
     * @return String
     */
    public static String urlDecode(String part) {
        try {
            if (StringUtils.isBlank(part)) {
                return null;
            }
            return URLDecoder.decode(part, DEFAULT_CHARSET);
        } catch (UnsupportedEncodingException e) {
            throw new BizException("urlDecode fail", e);
        }
    }

    /**
     * 获取字符串的字节数组
     *
     * @param content 字符串
     * @return byte[]
     */
    public static byte[] getBytes(String content) {
        return getBytes(content, StandardCharsets.UTF_8);
    }

    public static byte[] getBytes(String content, Charset charset) {
        try {
            return content.getBytes(charset.name());
        } catch (UnsupportedEncodingException e) {
            throw new BizException("getBytes fail", e);
        }
    }

    /**
     * 转换字节数组为字符串
     *
     * @param content 字节数组
     * @return String
     */
    public static String getString(byte[] content) {
        return getString(content, StandardCharsets.UTF_8);
    }

    /**
     * 指定编码方式转换字节数组为字符串
     *
     * @param content 字符串字节数组
     * @param charset 编码
     * @return String
     */
    public static String getString(byte[] content, Charset charset) {
        try {
            return new String(content, charset.name());
        } catch (UnsupportedEncodingException e) {
            throw new BizException("getString fail", e);
        }
    }

    /**
     * Base62编码
     *
     * @param content 密文
     * @return String
     */
    public static String encodeBase62(String content) {
        return Base62.encodeBase62(content);
    }

    /**
     * Base62解码
     *
     * @param content 密文
     * @return String
     */
    public static String decodeBase62(String content) {
        return Base62.decodeBase62(content);
    }

    /**
     * sha256_HMAC加密
     *
     * @param message 消息
     * @param secret  秘钥
     * @return 加密后字符串
     */
    public static String hmacSha256(String message, String secret) {
        try {
            final String algorithm = "HmacSHA256";
            Mac hmacSha256 = Mac.getInstance(algorithm);
            SecretKeySpec secretKey = new SecretKeySpec(EncodeUtils.getBytes(secret), algorithm);
            hmacSha256.init(secretKey);
            byte[] bytes = hmacSha256.doFinal(message.getBytes());
            return encodeHex(bytes);
        } catch (Exception e) {
            throw new BizException("hmac sha256 fail", e);
        }
    }
}
