package cn.wjee.commons.enums;

import cn.wjee.commons.lang.StringUtils;
import lombok.Getter;

import java.util.Arrays;

/**
 * 有效状态枚举（1-有效,0-无效）
 *
 * @author listening
 */
@Getter
public enum StatusEnum {
    /**
     * 有效
     */
    VALID(1, "有效"),
    /**
     * 无效
     */
    INVALID(0, "无效"),
    ;

    private final Integer code;
    private final String desc;

    StatusEnum(Integer code, String desc) {
        this.code = code;
        this.desc = desc;
    }

    public Byte getByteCode() {
        return Byte.parseByte(this.code + "");
    }

    /**
     * 根据编码查询枚举项
     *
     * @param code 编码
     * @return StatusEnum
     */
    public static StatusEnum getByCode(Byte code) {
        return getByCode(code + "");
    }

    public static StatusEnum getByCode(String code) {
        return Arrays.stream(StatusEnum.values()).filter(item -> item.isMatch(code)).findFirst().orElse(null);
    }

    public static StatusEnum getByCode(Integer code) {
        return Arrays.stream(StatusEnum.values()).filter(item -> item.isMatch(code)).findFirst().orElse(null);
    }

    /**
     * 给定编码，是否等于此枚举项
     *
     * @param code 编码值
     * @return boolean
     */
    public boolean isMatch(String code) {
        return StringUtils.isNotBlank(code) && StringUtils.equals(code, getCode() + "");
    }

    public boolean isMatch(Integer code) {
        return getCode().equals(code);
    }

    public boolean isMatch(Byte code) {
        return code != null && getCode().equals(Integer.parseInt(code + ""));
    }

    /**
     * 判断给定值是否在枚举中定义
     *
     * @param code 编码值
     * @return boolean
     */
    public static boolean isAny(String code) {
        return getByCode(code) != null;
    }

    public static boolean isAny(Integer code) {
        return code != null && isAny(code + "");
    }

    public static boolean isAny(Byte code) {
        return code != null && isAny(code + "");
    }

    /**
     * 根据编码查询名称
     *
     * @param code 编码
     * @return String
     */
    public static String getNameByCode(String code) {
        StatusEnum statusEnum = getByCode(code);
        return statusEnum != null ? statusEnum.getDesc() : null;
    }

    public static String getNameByCode(Byte code) {
        StatusEnum statusEnum = getByCode(code);
        return statusEnum != null ? statusEnum.getDesc() : null;
    }

    public static String getNameByCode(Integer code) {
        StatusEnum statusEnum = getByCode(code);
        return statusEnum != null ? statusEnum.getDesc() : null;
    }

    /**
     * 根据名字查询编码
     *
     * @param name 名字
     * @return String
     */
    public static String getCodeByName(String name) {
        if (StringUtils.isBlank(name)) {
            return null;
        }
        StatusEnum statusEnum = Arrays.stream(StatusEnum.values())
            .filter(item -> StringUtils.equalsIgnoreCase(item.getDesc(), name))
            .findFirst().orElse(null);
        return statusEnum != null ? statusEnum.getCode() + "" : null;
    }
}
