package cn.wjee.commons.http;

import cn.wjee.commons.enums.HttpHeaderEnum;
import cn.wjee.commons.enums.HttpMethodEnum;
import cn.wjee.commons.exception.BizException;
import cn.wjee.commons.exception.BusinessException;
import cn.wjee.commons.io.IOUtils;
import cn.wjee.commons.lang.StringUtils;
import lombok.AllArgsConstructor;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.ByteArrayInputStream;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.Proxy;
import java.net.URL;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Consumer;
import java.util.function.Supplier;

/**
 * Http请求构建
 *
 * @author listening
 */
public class RequestBuilder {
    /**
     * 日志
     */
    private static final Logger log = LoggerFactory.getLogger(RequestBuilder.class);

    /**
     * Cookie
     */
    protected static final String REQUEST_COOKIE = "Cookie";
    protected static final String RESPONSE_COOKIE_HEADER = "Set-Cookie";

    /**
     * 请求地址
     */
    private String url;
    /**
     * 请求内容
     */
    private String contentType;
    /**
     * 请求方式
     */
    private HttpMethodEnum httpMethod;
    /**
     * 编码方式,默认utf-8
     */
    private Charset encoding = StandardCharsets.UTF_8;
    /**
     * Connect Timeout
     */
    private Integer connectTimeout = 1000;
    /**
     * Read Timeout
     */
    private Integer readTimeout = 5000;
    /**
     * Http Headers
     */
    private final Map<String, String> headers = new ConcurrentHashMap<>();
    /**
     * Cookie
     */
    private final Map<String, Object> cookies = new ConcurrentHashMap<>();
    /**
     * 网络代理
     */
    private Proxy proxy;

    private Supplier<InputStream> bodySupplier;

    private RequestBuilder() {

    }

    public RequestBuilder url(String url) {
        this.url = url;
        return this;
    }

    public RequestBuilder media(String contentType, Charset encoding) {
        this.contentType = contentType;
        if (encoding != null) {
            this.encoding = encoding;
        }
        return this;
    }

    public RequestBuilder json(String content) {
        this.contentType = "application/json";
        this.encoding = StandardCharsets.UTF_8;
        this.bodySupplier = () -> new ByteArrayInputStream(content.getBytes(StandardCharsets.UTF_8));
        return this;
    }

    public RequestBuilder body(Supplier<InputStream> body) {
        this.bodySupplier = body;
        return this;
    }

    public RequestBuilder method(HttpMethodEnum httpMethod) {
        this.httpMethod = httpMethod;
        return this;
    }

    public RequestBuilder options(Integer connectTimeout, Integer readTimeout) {
        if (connectTimeout != null && connectTimeout > 0) {
            this.connectTimeout = connectTimeout;
        }
        if (readTimeout != null && readTimeout > 0) {
            this.readTimeout = readTimeout;
        }
        return this;
    }

    public RequestBuilder header(String name, String value) {
        this.headers.put(name, value);
        return this;
    }

    public RequestBuilder headers(Map<String, String> headers) {
        if (headers != null) {
            this.headers.putAll(headers);
        }
        return this;
    }

    public RequestBuilder cookie(String name, String value) {
        this.cookies.put(name, value);
        return this;
    }

    public RequestBuilder proxy(Proxy proxy) {
        this.proxy = proxy;
        return this;
    }

    public static RequestBuilder get(String url) {
        return new RequestBuilder().method(HttpMethodEnum.GET).url(url);
    }

    public static RequestBuilder post(String url) {
        return new RequestBuilder().method(HttpMethodEnum.POST).url(url);
    }

    protected void execute(Consumer<HttpResponse> response) {
        HttpURLConnection httpConnection = null;
        try (HttpResponse httpResponse = new HttpResponse()) {
            httpConnection = openConnection();
            setBody(httpConnection);
            // 连接
            httpConnection.connect();
            if (response != null) {
                httpResponse.setCode(httpConnection.getResponseCode());
                httpResponse.setBody(httpConnection.getInputStream());
                response.accept(httpResponse);
            }
        } catch (Exception e) {
            throw new BusinessException("execute fail", e);
        } finally {
            closeHttpConnection(httpConnection);
        }
    }

    private void setBody(HttpURLConnection httpConnection) {
        try {
            if (bodySupplier != null) {
                httpConnection.setDoOutput(true);
                httpConnection.setDoInput(true);
                IOUtils.transStream(bodySupplier.get(), httpConnection.getOutputStream());
            }
        } catch (IOException e) {
            log.error("HttpPost.body set request fail", e);
        }
    }

    private void closeHttpConnection(HttpURLConnection httpConnection) {
        try {
            if (httpConnection != null) {
                httpConnection.disconnect();
            }
        } catch (Exception e) {
            log.debug("closeHttpConnection fail", e);
        }

    }

    /**
     * 创建链接
     *
     * @return HttpURLConnection
     */
    private HttpURLConnection openConnection() {
        try {
            URL requestUrl = new URL(this.url);
            final HttpURLConnection connection;
            // 设置代理
            if (this.proxy != null) {
                connection = (HttpURLConnection) requestUrl.openConnection(this.proxy);
            } else {
                connection = (HttpURLConnection) requestUrl.openConnection();
            }
            connection.setRequestMethod(this.httpMethod.name());

            // 不允许交互
            connection.setAllowUserInteraction(false);
            // 不允许使用缓存
            connection.setDefaultUseCaches(false);
            // 不允许输入
            connection.setDoInput(true);
            // 允许输出
            connection.setDoOutput(true);

            // 设置超时时间
            connection.setConnectTimeout(this.connectTimeout);
            connection.setReadTimeout(this.readTimeout);
            // 设置请求头
            connection.setRequestProperty("connection", "keep-alive");
            connection.setRequestProperty("Charset", encoding.name());
            if (StringUtils.isNotBlank(contentType)) {
                connection.setRequestProperty(HttpHeaderEnum.CONTENT_TYPE.getCode(), contentType);
            }
            headers.forEach(connection::setRequestProperty);

            // 设置Cookie
            StringBuilder cookieBuilder = new StringBuilder();
            cookies.forEach((key, value) -> cookieBuilder.append(key).append("=").append(value).append(";"));
            String cookie = cookieBuilder.toString();
            if (StringUtils.isNotBlank(cookie)) {
                connection.setRequestProperty(REQUEST_COOKIE, encoding.name());
            }
            return connection;
        } catch (IOException e) {
            throw new BizException("openConnection fail", e);
        }
    }

    public HttpResponse execute() {
        HttpResponse response = new HttpResponse();
        try {
            execute(tempRes -> {
                response.setCode(tempRes.code);
                response.setBody(new ByteArrayInputStream(IOUtils.toByteArray(tempRes.getBody())));
            });
        } catch (Exception e) {
            response.setCode(400);
        }
        return response;
    }

    @Getter
    @Setter
    @AllArgsConstructor
    @NoArgsConstructor
    public static class HttpResponse implements Closeable {
        // Status Code
        private Integer code;
        // Body InputStream
        private InputStream body;

        @Override
        public void close() {
            IOUtils.closeQuietly(body);
        }
    }
}
