package cn.wjee.commons.io;

import cn.wjee.commons.collection.CollectionUtils;
import cn.wjee.commons.constants.Vars;
import cn.wjee.commons.crypto.EncodeUtils;
import cn.wjee.commons.exception.Asserts;
import cn.wjee.commons.exception.BizException;
import cn.wjee.commons.exception.BusinessException;
import cn.wjee.commons.lang.StringUtils;
import lombok.SneakyThrows;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.stream.Collectors;

/**
 * 文件工具类
 *
 * @author listening
 */
public class FileUtils {
    /**
     * 日志
     */
    private static final Logger log = LoggerFactory.getLogger(FileUtils.class);

    private FileUtils() {

    }

    /**
     * 读取文件内容
     *
     * @param fileName 文件路径
     * @return byte[]
     */
    @SneakyThrows
    public static byte[] readFileToBytes(String fileName) {
        return IOUtils.toByteArray(new FileInputStream(fileName));
    }

    /**
     * 字节数组写文件
     *
     * @param fileName 文件路径
     * @param data     字节数据
     */
    public static void writeBytesToFile(String fileName, byte[] data) {
        File saveFile = new File(fileName);
        File parentDirectory = saveFile.getParentFile();
        if (!parentDirectory.exists()) {
            boolean mkdirsResult = parentDirectory.mkdirs();
            Asserts.isTrue(mkdirsResult, "创建目录失败");
        }
        if (!parentDirectory.isDirectory() || !parentDirectory.canWrite()) {
            throw new BizException("保存目录不可操作");
        }
        try (FileOutputStream out = new FileOutputStream(saveFile)) {
            out.write(data);
            out.flush();
        } catch (Exception e) {
            throw new BizException("写文件失败", e);
        }
    }

    /**
     * 创建目录
     *
     * @param path 目录
     */
    public static void mkdirs(String path) {
        File directory = new File(path);
        if (!directory.exists()) {
            boolean mkdirsResult = directory.mkdirs();
            Asserts.isTrue(mkdirsResult, "创建目录失败");
        }
    }

    /**
     * 删除文件或目录
     *
     * @param file 文件
     */
    public static void delete(File file) {
        try {
            if (!file.exists()) {
                return;
            }
            if (!file.isDirectory()) {
                Files.delete(file.toPath());
                return;
            }

            File[] files = file.listFiles();
            if (files == null || files.length == 0) {
                Files.delete(file.toPath());
                return;
            }

            for (File tempFile : files) {
                if (tempFile.isDirectory()) {
                    delete(tempFile);
                } else {
                    Files.delete(tempFile.toPath());
                }
            }
            Files.delete(file.toPath());
        } catch (IOException e) {
            log.error("delete file fail", e);
        }
    }

    /**
     * 列举文件夹下的所有文件以及文件夹
     *
     * @param directory 目录
     * @param predicate 文件过滤
     * @param recursive 是否递归查询
     * @param returnDir 是否返回文件夹
     * @return List
     */
    public static List<File> listFiles(File directory, Predicate<File> predicate, boolean returnDir, boolean recursive) {
        List<File> resultList = new ArrayList<>();
        if (!directory.exists()) {
            return resultList;
        }
        File[] files = directory.listFiles();
        if (files == null || files.length == 0) {
            return resultList;
        }
        if (predicate == null) {
            predicate = file -> true;
        }
        List<File> collectList = Arrays.stream(files).filter(predicate).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(collectList)) {
            return resultList;
        }
        for (File file : collectList) {
            if (file.isDirectory()) {
                if (returnDir) {
                    resultList.add(file);
                }
                if (recursive) {
                    resultList.addAll(listFiles(file, predicate, returnDir, true));
                }
            } else {
                resultList.add(file);
            }
        }
        return resultList;
    }

    /**
     * 遍历目录
     *
     * @param directory 目录
     * @param consumer  文件回调
     * @param recursive 是否递归查询
     */
    public static void trace(File directory, Consumer<File> consumer, boolean recursive) {
        if (!directory.exists()) {
            return;
        }
        if (directory.isFile()) {
            consumer.accept(directory);
            return;
        }
        File[] files = directory.listFiles();
        if (files == null || files.length == 0) {
            return;
        }
        for (File file : files) {
            if (file.isDirectory()) {
                consumer.accept(file);
                if (recursive) {
                    trace(file, consumer, true);
                }
            } else {
                consumer.accept(file);
            }
        }
    }

    /**
     * 读取文件内容到列表
     *
     * @param file    文件路径
     * @param charset 文件编码
     * @return List
     */
    public static List<String> readLines(String file, Charset charset) {
        List<String> result = new ArrayList<>();
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(new FileInputStream(file), charset))) {
            String line;
            while ((line = reader.readLine()) != null) {
                result.add(line);
            }
        } catch (Exception e) {
            throw new BusinessException("readLines fail", e);
        }
        return result;
    }

    /**
     * 读取文件内容到列表
     *
     * @param file     文件路径
     * @param charset  文件编码
     * @param consumer 消费回调
     */
    public static void consumeLines(String file, Charset charset, Consumer<String> consumer) {
        if (StringUtils.isBlank(file) || consumer == null) {
            return;
        }
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(new FileInputStream(file), charset))) {
            String line;
            while ((line = reader.readLine()) != null) {
                consumer.accept(line);
            }
        } catch (Exception e) {
            throw new BusinessException("readLines fail", e);
        }
    }

    /**
     * 写文件
     *
     * @param filePath     文件路径
     * @param charSequence 内容
     */
    public static void write(String filePath, CharSequence charSequence) {
        write(filePath, charSequence, StandardCharsets.UTF_8, false);
    }

    /**
     * 写文件
     *
     * @param filePath     文件路径
     * @param charSequence 内容
     * @param append       追加
     */
    public static void write(String filePath, CharSequence charSequence, boolean append) {
        write(filePath, charSequence, StandardCharsets.UTF_8, append);
    }


    /**
     * 写文件
     *
     * @param filePath     文件路径
     * @param charset      文件编码
     * @param charSequence 内容
     * @param append       追加
     */
    public static void write(String filePath, CharSequence charSequence, Charset charset, boolean append) {
        File file = new File(filePath);
        File parent = file.getParentFile();
        if (!parent.exists() && !parent.mkdirs()) {
            log.warn("mkdir fail : {}", parent.getPath());
        }
        try (BufferedOutputStream out = new BufferedOutputStream(new FileOutputStream(file, append))) {
            String data = charSequence == null ? "" : charSequence.toString();
            out.write(EncodeUtils.getBytes(data, charset));
        } catch (Exception e) {
            throw new BusinessException("write fail", e);
        }
    }

    /**
     * 获取文件后缀名
     *
     * @param fileName 值
     * @param withDot  是否包含点
     * @return String
     */
    public static String getSuffix(String fileName, boolean withDot) {
        if (StringUtils.isBlank(fileName) || !fileName.contains(Vars.DOT)) {
            return null;
        }
        return fileName.substring(fileName.lastIndexOf(Vars.DOT) + (withDot ? 0 : 1));
    }

    /**
     * 获取文件前缀名
     *
     * @param fileName 文件名
     * @return String
     */
    public static String getPrefix(String fileName) {
        if (StringUtils.isBlank(fileName) || !fileName.contains(Vars.DOT)) {
            return null;
        }
        return fileName.substring(0, fileName.lastIndexOf(Vars.DOT));
    }

    /**
     * 获取文件名
     *
     * @param path 地址
     * @return String
     */
    public static String getFileName(String path) {
        if (StringUtils.isBlank(path)) {
            return null;
        }
        String destVal = path;
        if (StringUtils.contains(destVal, Vars.QUESTION)) {
            destVal = destVal.substring(0, destVal.indexOf(Vars.QUESTION));
        }
        if (StringUtils.contains(destVal, Vars.SEPARATOR)) {
            destVal = StringUtils.substring(destVal, destVal.lastIndexOf(Vars.SEPARATOR) + 1);
        }
        if (StringUtils.contains(destVal, Vars.WIN_FILE_SEPARATOR)) {
            destVal = StringUtils.substring(destVal, destVal.lastIndexOf(Vars.WIN_FILE_SEPARATOR) + 1);
        }
        return destVal;
    }
}
