package cn.wjee.commons.io;

import cn.wjee.commons.crypto.EncodeUtils;
import cn.wjee.commons.exception.Asserts;
import cn.wjee.commons.exception.BizException;
import cn.wjee.commons.exception.BusinessException;
import cn.wjee.commons.lang.StringUtils;
import lombok.extern.slf4j.Slf4j;

import java.io.*;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;

/**
 * IO工具类
 */
@Slf4j
public class IOUtils {

    private IOUtils() {

    }

    /**
     * 关闭流
     *
     * @param closeable Closeable
     */
    public static void closeQuietly(Closeable closeable) {
        try {
            if (closeable != null) {
                closeable.close();
            }
        } catch (IOException e) {
            log.error("closeQuietly fail", e);
        }
    }

    /**
     * 读取输入流到字节数组
     *
     * @param inputStream 输入流
     * @return byte[]
     */
    public static byte[] toByteArray(InputStream inputStream) {
        try (ByteArrayOutputStream out = new ByteArrayOutputStream()) {
            if (inputStream == null) {
                return new byte[0];
            }
            byte[] buffer = new byte[2048];
            int index;
            while ((index = inputStream.read(buffer)) != -1) {
                out.write(buffer, 0, index);
            }
            return out.toByteArray();
        } catch (Exception e) {
            throw new BusinessException("toByteArray fail", e);
        }
    }

    /**
     * 输入流转字符串
     *
     * @param inputStream 输入流
     * @param charset     编码
     * @return String
     */
    public static String toString(InputStream inputStream, Charset charset) {
        byte[] bytes = toByteArray(inputStream);
        return EncodeUtils.getString(bytes, charset);
    }

    /**
     * 输入流转字符串
     *
     * @param inputStream 输入流
     * @return String
     */
    public static String toString(InputStream inputStream) {
        return toString(inputStream, StandardCharsets.UTF_8);
    }

    /**
     * 字符串转输出流
     *
     * @param data    字符串
     * @param charset 编码 StandardCharsets.UTF_8
     * @return String
     */
    public static InputStream toInputStream(String data, Charset charset) {
        return new ByteArrayInputStream(EncodeUtils.getBytes(data, charset));
    }

    /**
     * 根据ClassLoader读取文件内容
     *
     * @param resourcePath 资源路径
     * @param classLoader  加载器
     * @return String
     */
    public static String getResourceAsString(ClassLoader classLoader, String resourcePath) {
        InputStream resourceInputStream = null;
        try {
            resourceInputStream = classLoader.getResourceAsStream(resourcePath);
            return new String(toByteArray(resourceInputStream), StandardCharsets.UTF_8);
        } finally {
            closeQuietly(resourceInputStream);
        }
    }

    /**
     * Copy资源
     *
     * @param src  源文件
     * @param dest 目标文件
     */
    public static void copyResource(String src, String dest) {
        Asserts.isTrue(!StringUtils.isAnyBlank(src, dest), "源文件和目标文件信息不能为空");
        File srcFile = new File(src);
        if (!srcFile.exists()) {
            throw new BizException("源文件不存在");
        }
        try (FileInputStream fileInputStream = new FileInputStream(srcFile)) {
            copyStream(fileInputStream, new File(dest));
        } catch (IOException e) {
            throw new BizException("复制文件失败", e);
        }
    }

    /**
     * 保存输入流到文件
     *
     * @param input    输入流
     * @param destFile 目标文件
     */
    public static void copyStream(InputStream input, File destFile) {
        if (destFile == null) {
            throw new BizException("目标文件不能为空");
        }

        File destParentFile = destFile.getParentFile();
        if (!destParentFile.exists()) {
            boolean mkDirs = destParentFile.mkdirs();
            Asserts.isTrue(mkDirs, "目标文件存储目录创建失败");
        }
        if (!destParentFile.canWrite()) {
            throw new BizException("目标文件夹无写权限");
        }
        try (FileOutputStream outputStream = new FileOutputStream(destFile)) {
            transStream(input, outputStream);
        } catch (IOException e) {
            throw new BizException("copyStream失败", e);
        }
    }

    /**
     * 保存输入流到输出流
     *
     * @param inputStream  输入流
     * @param outputStream 输出流
     */
    public static void transStream(InputStream inputStream, OutputStream outputStream) {
        if (inputStream == null || outputStream == null) {
            return;
        }
        try (
            BufferedInputStream bufferedInputStream = new BufferedInputStream(inputStream);
            BufferedOutputStream bufferedOutputStream = new BufferedOutputStream(outputStream)) {
            byte[] buffer = new byte[2048];
            int len;
            while ((len = bufferedInputStream.read(buffer)) != -1) {
                bufferedOutputStream.write(buffer, 0, len);
            }
            bufferedOutputStream.flush();
        } catch (IOException e) {
            throw new BizException("转流失败", e);
        }
    }
}
