package cn.wjee.commons.io;

import cn.wjee.commons.constants.Vars;
import cn.wjee.commons.exception.Asserts;
import cn.wjee.commons.exception.BizException;
import cn.wjee.commons.functional.KeyValueConsumer;
import cn.wjee.commons.lang.StringUtils;
import lombok.AccessLevel;
import lombok.Cleanup;
import lombok.NoArgsConstructor;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.function.Predicate;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipOutputStream;

/**
 * Zip解压缩工具
 *
 * @author listening
 */
@Slf4j
@NoArgsConstructor(access = AccessLevel.PRIVATE)
public class ZipUtils {

    /**
     * 复制Jar包中的资源文件到指定目录
     *
     * @param jarPath            JAR路径
     * @param savePath           保存目录
     * @param removeEntryPackage 是否清除Entry的包路径
     */
    public static void copy(File jarPath, String savePath, boolean removeEntryPackage) {
        try (JarFile jarFile = new JarFile(jarPath)) {
            File saveDirectory = new File(savePath);
            if (!saveDirectory.exists()) {
                boolean result = saveDirectory.mkdirs();
                Asserts.isTrue(result, "创建文件存储目录失败");
            }
            Asserts.isTrue(saveDirectory.isDirectory(), "保存路径必须为文件夹");

            Enumeration<JarEntry> entries = jarFile.entries();
            while (entries.hasMoreElements()) {
                JarEntry jarEntry = entries.nextElement();
                String entryName = jarEntry.getName();
                if (jarEntry.isDirectory()) {
                    continue;
                }
                String simpleEntryName = entryName.substring(entryName.lastIndexOf(Vars.SEPARATOR) + 1);
                String saveName = removeEntryPackage ? simpleEntryName : entryName;
                try (InputStream inputStream = jarFile.getInputStream(jarEntry)) {
                    IOUtils.copyStream(inputStream, new File(savePath + Vars.SEPARATOR + saveName));
                }
            }
        } catch (IOException e) {
            throw new BizException("Jar Resources Copy Fail", e);
        }
    }

    /**
     * 解压ZIP文件
     *
     * @param zipFile   ZIP文件
     * @param exactPath 解压路径
     * @throws IOException ex
     */
    public static void unzip(File zipFile, String exactPath) throws IOException {
        if (zipFile == null) {
            return;
        }
        String originFileName = FileUtils.getSuffix(zipFile.getName(), false);
        Asserts.isTrue(StringUtils.equalsIgnoreCase(originFileName, "zip"), "不是有效的ZIP文件");
        // 解压目录
        if (StringUtils.isBlank(exactPath)) {
            exactPath = zipFile.getParent();
        }
        File exactDirectory = new File(exactPath);
        if (!exactDirectory.exists()) {
            boolean mkdir = exactDirectory.mkdirs();
            Asserts.isTrue(mkdir, "目录创建失败");
        }
        try (ZipFile zip = new ZipFile(zipFile)) {
            Enumeration<? extends ZipEntry> entries = zip.entries();
            while (entries.hasMoreElements()) {
                ZipEntry zipEntry = entries.nextElement();
                String entryName = zipEntry.getName();
                String savePath = exactDirectory.getAbsolutePath() + Vars.SEPARATOR + entryName;
                if (zipEntry.isDirectory()) {
                    FileUtils.mkdirs(savePath);
                    continue;
                }
                // 文件
                try (InputStream inputStream = zip.getInputStream(zipEntry)) {
                    IOUtils.copyStream(inputStream, new File(savePath));
                }
            }
        }
    }

    /**
     * ZIP压缩
     *
     * @param src      源文件
     * @param savePath 保存目录
     * @param saveName 保存名称
     */
    @SneakyThrows
    public static void zip(String src, String savePath, String saveName) {
        try (ZipOutputStream zipOut = new ZipOutputStream(Files.newOutputStream(Paths.get(savePath + "/" + saveName)))) {
            FileUtils.trace(new File(src), file -> {
                try {
                    String relativePath = StringUtils.substring(file.getAbsolutePath(), src.length());
                    // 文件夹
                    if (file.isDirectory()) {
                        ZipEntry zipEntry = new ZipEntry(relativePath + "/");
                        zipOut.putNextEntry(zipEntry);
                        zipOut.closeEntry();
                    }
                    // 文件
                    else {
                        ZipEntry zipEntry = new ZipEntry(relativePath);
                        zipOut.putNextEntry(zipEntry);
                        @Cleanup BufferedInputStream inputStream = new BufferedInputStream(Files.newInputStream(file.toPath()));
                        byte[] buffer = new byte[1024];
                        int index;
                        while ((index = inputStream.read(buffer)) != -1) {
                            zipOut.write(buffer, 0, index);
                        }
                        zipOut.closeEntry();
                    }
                } catch (Exception e) {
                    log.error("zip FileUtils.trace fail", e);
                }
            }, true);
            zipOut.finish();
        }
    }


    /**
     * ZIP截取压缩
     *
     * @param src     源文件
     * @param isFlat  是否扁平化(文件位于根目录)
     * @param saveFie 保存目录
     * @param filter  筛选
     */
    @SneakyThrows
    public static void subZip(File src, String saveFie, boolean isFlat, Predicate<ZipEntry> filter) {
        try (ZipOutputStream zipOut = new ZipOutputStream(Files.newOutputStream(Paths.get(saveFie)));
             ZipFile zip = new ZipFile(src)) {
            Enumeration<? extends ZipEntry> entries = zip.entries();
            while (entries.hasMoreElements()) {
                ZipEntry zipEntry = entries.nextElement();
                if (filter == null || !filter.test(zipEntry)) {
                    continue;
                }
                // 文件夹
                if (zipEntry.isDirectory()) {
                    ZipEntry tempEntry = new ZipEntry(zipEntry.getName());
                    zipOut.putNextEntry(tempEntry);
                }
                // 文件
                else {
                    String fileName = zipEntry.getName();
                    if (isFlat) {
                        String tempName = FileUtils.getFileName(fileName);
                        fileName = StringUtils.getValue(tempName, fileName);
                    }
                    ZipEntry tempEntry = new ZipEntry(fileName);
                    zipOut.putNextEntry(tempEntry);
                    @Cleanup BufferedInputStream inputStream = new BufferedInputStream(zip.getInputStream(zipEntry));
                    byte[] buffer = new byte[1024];
                    int index;
                    while ((index = inputStream.read(buffer)) != -1) {
                        zipOut.write(buffer, 0, index);
                    }
                }
                zipOut.closeEntry();
                zipOut.finish();
            }
        }
    }

    /**
     * 列举ZIP压缩包文件
     *
     * @param zipFilePath 压缩包文件
     * @param consumer    回调
     */
    public static void listZip(File zipFilePath, KeyValueConsumer<ZipFile, ZipEntry> consumer) {
        if (zipFilePath == null || !zipFilePath.exists()) {
            return;
        }
        try (ZipFile zip = new ZipFile(zipFilePath)) {
            Enumeration<? extends ZipEntry> entries = zip.entries();
            while (entries.hasMoreElements()) {
                ZipEntry zipEntry = entries.nextElement();
                consumer.accept(zip, zipEntry);
            }
        } catch (IOException e) {
            throw new BizException("文件读取失败", e);
        }
    }

    /**
     * 列举ZIP压缩包文件
     *
     * @param zipFilePath 压缩包文件
     * @param filter      筛选断言
     * @param consumer    回调
     */
    public static void listZip(File zipFilePath, Predicate<ZipEntry> filter, KeyValueConsumer<ZipFile, List<ZipEntry>> consumer) {
        if (zipFilePath == null || !zipFilePath.exists()) {
            return;
        }
        try (ZipFile zip = new ZipFile(zipFilePath)) {
            Enumeration<? extends ZipEntry> entries = zip.entries();

            final List<ZipEntry> filteredEntryList = new ArrayList<>();
            while (entries.hasMoreElements()) {
                ZipEntry zipEntry = entries.nextElement();
                if (filter != null && filter.test(zipEntry)) {
                    filteredEntryList.add(zipEntry);
                }
            }
            if (consumer != null) {
                consumer.accept(zip, filteredEntryList);
            }
        } catch (IOException e) {
            throw new BizException("文件读取失败", e);
        }
    }
}
