package cn.wjee.commons.lang;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.ZoneId;
import java.util.Calendar;
import java.util.Date;
import java.util.TimeZone;

/**
 * 日期工具
 *
 * @author listening
 */
public class DateUtils {
    private DateUtils() {

    }

    public static final String TIME_ZONE_GMT8 = "GMT+8";
    public static final TimeZone DEFAULT_TIME_ZONE = TimeZone.getTimeZone(TIME_ZONE_GMT8);

    public static final String FORMAT_DATE = "yyyy-MM-dd";
    public static final String FORMAT_DATE_MONTH = "yyyy-MM";
    public static final String FORMAT_DATE_YEAR = "yyyy";
    public static final String FORMAT_TIME = "HH:mm:ss";
    public static final String FORMAT_TIME_MINUTE = "HH:mm";
    public static final String FORMAT_TIME_HOUR = "HH";
    public static final String FORMAT_DATETIME = "yyyy-MM-dd HH:mm:ss";
    public static final String FORMAT_TIMESTAMP = "yyyyMMddHHmmsssss";

    /**
     * Format Date Time
     *
     * @param date   日期
     * @param format 格式
     * @return String
     */
    public static String format(Date date, String format) {
        if (date == null || StringUtils.isBlank(format)) {
            return null;
        }
        SimpleDateFormat simpleDateFormat = new SimpleDateFormat(format);
        simpleDateFormat.setTimeZone(DEFAULT_TIME_ZONE);
        return simpleDateFormat.format(date);
    }

    /**
     * 解析日期
     *
     * @param date   日期
     * @param format 格式
     * @return Date
     */
    public static Date parse(String date, String format) {
        try {
            if (StringUtils.isAnyBlank(date, format)) {
                return null;
            }
            SimpleDateFormat simpleDateFormat = new SimpleDateFormat(format);
            simpleDateFormat.setTimeZone(DEFAULT_TIME_ZONE);
            return simpleDateFormat.parse(date);
        } catch (ParseException e) {
            return null;
        }
    }

    /**
     * Format Date Time
     *
     * @param date 日期
     * @return String
     */
    public static String datetime2string(Date date) {
        return format(date, FORMAT_DATETIME);
    }

    /**
     * Format Date Time
     *
     * @param date 日期
     * @return String
     */
    public static String date2string(Date date) {
        return format(date, FORMAT_DATE);
    }

    /**
     * String to Date
     *
     * @param value 字符串日期
     * @return Date
     */
    public static Date string2date(String value) {
        return parse(value, FORMAT_DATE);
    }

    /**
     * 时间加减
     *
     * @param date  开始日期
     * @param num   增加数
     * @param field 增加单位
     * @return Date
     */
    public static Date add(Date date, int num, int field) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.add(field, num);
        return calendar.getTime();
    }

    /**
     * 时间加减天数
     *
     * @param date 开始日期
     * @param num  增加数
     * @return Date
     */
    public static Date addDay(Date date, int num) {
        return add(date, num, Calendar.DATE);
    }

    /**
     * 日期间隔维度
     *
     * @param start         开始
     * @param end           结束
     * @param calendarField 维度
     * @return int
     */
    public static int getDateBetween(Date start, Date end, Integer calendarField) {
        if (start == null) {
            return 0;
        }
        if (end == null) {
            end = new Date();
        }
        int between = 0;
        Calendar startCalendar = Calendar.getInstance();
        startCalendar.setTime(start);

        Calendar endCalendar = Calendar.getInstance();
        endCalendar.setTime(end);

        while (startCalendar.before(endCalendar)) {
            between += 1;
            startCalendar.add(calendarField, 1);
        }
        return startCalendar.after(endCalendar) ? (between - 1) : between;
    }

    /**
     * 日期间隔天数
     *
     * @param start 开始
     * @param end   结束
     * @return int
     */
    public static int getBetweenDays(Date start, Date end) {
        return getDateBetween(start, end, Calendar.DATE);
    }

    /**
     * 日期间隔年数
     *
     * @param start 开始
     * @param end   结束
     * @return int
     */
    public static int getBetweenYears(Date start, Date end) {
        return getDateBetween(start, end, Calendar.YEAR);
    }

    /**
     * 是否日期相同
     *
     * @param one    日期1
     * @param two    日期2
     * @param format 格式
     * @return boolean
     */
    public static boolean isSame(Date one, Date two, String format) {
        if (one == null || two == null || StringUtils.isBlank(format)) {
            return false;
        }
        return StringUtils.equals(format(one, format), format(two, format));
    }

    /**
     * 获取时间戳（精确到毫秒）
     *
     * @return String
     */
    public static String getTimestamp() {
        return format(new Date(), FORMAT_TIMESTAMP);
    }

    /**
     * 倒计时
     *
     * @param endDate 截止日
     * @return String
     */
    public static String timeCountDown(Date endDate) {
        Date now = new Date();
        if (endDate.before(now)) {
            return "";
        }
        long midTime = (endDate.getTime() - System.currentTimeMillis()) / 1000;
        return (midTime / 60 / 60 / 24) + "天"
            + (midTime / 60 / 60 % 24) + "时"
            + (midTime / 60 % 60) + "分"
            + (midTime % 60) + "秒";
    }

    /**
     * 获取时间戳（秒）
     *
     * @param date 日期
     * @return Long
     */
    public static Long getUnixTime(Date date) {
        return date == null ? null : date.getTime() / 1000;
    }

    /**
     * 获取时间戳（秒）
     *
     * @param date 日期
     * @return Long
     */
    public static Date getOnlyDate(Date date) {
        Calendar calendar = Calendar.getInstance(DEFAULT_TIME_ZONE);
        calendar.setTime(date);
        calendar.set(Calendar.HOUR_OF_DAY, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        calendar.set(Calendar.MILLISECOND, 0);
        return calendar.getTime();
    }

    /**
     * 获取当日日期
     *
     * @return String
     */
    public static String getLocalDate() {
        return LocalDate.now(ZoneId.of(TIME_ZONE_GMT8)).toString();
    }
}
