package cn.wjee.commons.lang;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.List;

/**
 * Number Utils
 *
 * @author wjee
 */
public class DecimalUtils {
    private DecimalUtils() {

    }

    /**
     * 数字100
     */
    public static final BigDecimal ONE_HUNDRED = new BigDecimal("100");

    /**
     * Convert String to BigDecimal
     *
     * @param value 值
     * @return BigDecimal
     */
    public static BigDecimal getDecimal(String value) {
        if (StringUtils.isBlank(value)) {
            return BigDecimal.ZERO;
        }
        if (!RegExpUtils.isWildDecimal(value.trim())) {
            return BigDecimal.ZERO;
        }
        return new BigDecimal(value.trim());
    }

    /**
     * validate the value between min and max
     *
     * @param value 值
     * @param min   最小
     * @param max   最大
     * @return boolean
     */
    public static boolean size(String value, String min, String max) {
        BigDecimal bdValue = getDecimal(value);
        BigDecimal bdMin = getDecimal(min);
        BigDecimal bdMax = getDecimal(max);
        return gte(bdValue, bdMin) && lte(bdValue, bdMax);
    }

    /**
     * 是否大于
     *
     * @param one 被减数
     * @param two 减数
     * @return boolean
     */
    public static boolean gt(BigDecimal one, BigDecimal two) {
        return one.compareTo(two) > 0;
    }

    /**
     * 是否小于
     *
     * @param one 数值1
     * @param two 数值2
     * @return boolean
     */
    public static boolean lt(BigDecimal one, BigDecimal two) {
        return one.compareTo(two) < 0;
    }

    /**
     * 是否大于等于
     *
     * @param one 数值1
     * @param two 数值2
     * @return boolean
     */
    public static boolean gte(BigDecimal one, BigDecimal two) {
        return one.compareTo(two) >= 0;
    }

    /**
     * 是否小于等于
     *
     * @param one 数值1
     * @param two 数值2
     * @return boolean
     */
    public static boolean lte(BigDecimal one, BigDecimal two) {
        return one.compareTo(two) <= 0;
    }

    /**
     * 按照金额格式化(2位小数)
     *
     * @param decimal 金额
     * @return String
     */
    public static String formatMoney(BigDecimal decimal) {
        return formatDecimal(decimal, 2);
    }

    /**
     * 按精度四色五入格式化位字符串
     *
     * @param decimal 数值
     * @param scale   有效精度
     * @return String
     */
    public static String formatDecimal(BigDecimal decimal, int scale) {
        BigDecimal tempValue = decimal == null ? BigDecimal.ZERO : decimal;
        return tempValue.setScale(scale, RoundingMode.HALF_UP).stripTrailingZeros().toString();
    }

    /**
     * 任一匹配(相等)
     *
     * @param dataList 匹配源
     * @param compare  匹配项
     * @return boolean
     */
    public static boolean isAnyEq(List<BigDecimal> dataList, BigDecimal compare) {
        if (dataList == null || compare == null) {
            return false;
        }
        return dataList.stream().anyMatch(item -> item != null && item.compareTo(compare) == 0);
    }

    /**
     * 金额从分转换元
     *
     * @param centAmount 金额（分）
     * @return BigDecimal
     */
    public static BigDecimal fromCent(Integer centAmount) {
        if (centAmount == null) {
            return BigDecimal.ZERO;
        }
        return new BigDecimal(centAmount + "").divide(ONE_HUNDRED, 2, RoundingMode.HALF_UP);
    }

    /**
     * 转换为分
     *
     * @param amount 金额
     * @return Integer
     */
    public static Integer getCent(BigDecimal amount) {
        return amount != null ? amount.multiply(ONE_HUNDRED).intValue() : 0;
    }
}
