package cn.wjee.commons.lang;

import cn.wjee.commons.exception.BizException;

import java.lang.annotation.Annotation;
import java.lang.reflect.*;
import java.util.*;
import java.util.stream.Collectors;

/**
 * Java Reflect Utils
 *
 * @author wjee
 * @version $Id: ReflectUtils.java, v 0.1 2015年6月11日 下午10:36:56 wjee Exp $
 */
@SuppressWarnings("all")
public class ReflectUtils {
    private ReflectUtils() {
    }

    /**
     * Java动态添加library path
     *
     * @param path 路径
     */
    public static void addLibraryPath(String path) {
        try {
            String libraryPath = System.getProperty("java.library.path");
            System.setProperty("java.library.path", libraryPath + ";" + path);
            Field fieldSysPath = ClassLoader.class.getDeclaredField("sys_paths");
            fieldSysPath.setAccessible(true);
            fieldSysPath.set(null, null);
        } catch (IllegalArgumentException | IllegalAccessException | NoSuchFieldException | SecurityException e) {
            throw new BizException("add library path fail");
        }
    }

    /**
     * Java动态添加library path
     */
    public static void addLibrarySigarPath() {
        String path = Objects.requireNonNull(ReflectUtils.class.getResource("/libsigar")).getPath();
        addLibraryPath(path);
    }

    /**
     * 获取泛型参数Class
     *
     * @param cls 类
     * @return Class
     */
    public static Class<?> getGenericClass(Class<?> cls) {
        if (cls == null || cls.getGenericSuperclass() == null) {
            return null;
        }
        Type type = cls.getGenericSuperclass();
        if (!(type instanceof ParameterizedType)) {
            return null;
        }
        ParameterizedType parameterType = (ParameterizedType) type;
        return (Class<?>) parameterType.getActualTypeArguments()[0];
    }

    /**
     * 泛型参数名称
     *
     * @param cls Class类
     * @return String
     */
    public static String getGenericClassName(Class<?> cls) {
        Class<?> genericClass = getGenericClass(cls);
        return genericClass == null ? null : genericClass.getSimpleName();
    }

    /**
     * 类型是否一致
     *
     * @param sourceClz  源Class
     * @param compareClz 目标Class
     * @return boolean
     */
    public static boolean isType(Class<?> sourceClz, Class<?> compareClz) {
        return sourceClz != null && compareClz != null && sourceClz.isAssignableFrom(compareClz);
    }

    /**
     * 根据Class实例一个对象
     *
     * @param clz 类Class
     * @param <T> 泛型Class
     * @return T
     */
    public static <T> T safeNewInstance(Class<T> clz) {
        try {
            return clz.getConstructor().newInstance();
        } catch (Exception e) {
            throw new BizException("Construct Clz Fail", e);
        }
    }

    /**
     * Find Annotation Fields
     *
     * @param clz             类Class
     * @param annotationClass 注解Class
     * @return List
     */
    public static List<Field> findAnnotationFields(Class<?> clz, Class<? extends Annotation> annotationClass) {
        return Arrays.stream(clz.getDeclaredFields())
            .filter(item -> item.isAnnotationPresent(annotationClass))
            .collect(Collectors.toList());
    }

    /**
     * 是否任一注解出现
     *
     * @param annotatedElement 方法
     * @param classes          注解数组
     * @return boolean
     */
    @SafeVarargs
    public static boolean isAnyAnnotationPresent(AnnotatedElement annotatedElement, Class<? extends Annotation>... classes) {
        if (classes == null) {
            return false;
        }
        for (Class<? extends Annotation> clz : classes) {
            if (annotatedElement.isAnnotationPresent(clz)) {
                return true;
            }
        }
        return false;
    }

    /**
     * 是否任一注解出现
     *
     * @param annotation        Annotation
     * @param annotatedElements 类型数组
     * @return boolean
     */
    public static boolean isAnyAnnotationPresent(Class<? extends Annotation> annotation, AnnotatedElement... annotatedElements) {
        if (annotatedElements == null) {
            return false;
        }
        for (AnnotatedElement element : annotatedElements) {
            if (element.isAnnotationPresent(annotation)) {
                return true;
            }
        }
        return false;
    }

    /**
     * 判断对象是否为数组
     *
     * @param type 类型
     * @param obj  对象
     * @return boolean
     */
    public static boolean isArray(Class<?> type, Object obj) {
        return (obj != null && obj.getClass().isArray()) || (type != null && type.isArray());
    }

    /**
     * 判断对象是否为Map
     *
     * @param type 类型
     * @param obj  对象
     * @return boolean
     */
    public static boolean isMap(Class<?> type, Object obj) {
        return type != null && Map.class.isAssignableFrom(type) || obj instanceof Map;
    }

    /**
     * 判断对象是否为列表
     *
     * @param type 类型
     * @param obj  对象
     * @return boolean
     */
    public static boolean isList(Class<?> type, Object obj) {
        return type != null && List.class.isAssignableFrom(type) || obj instanceof List;
    }

    /**
     * 判断对象是否为列表
     *
     * @param type 类型
     * @param obj  对象
     * @return boolean
     */
    public static boolean isSet(Class<?> type, Object obj) {
        return type != null && Set.class.isAssignableFrom(type) || obj instanceof Set;
    }

    /**
     * 判断对象是否为常见基本类型（String特殊处理）
     *
     * @param obj           对象
     * @param includeString 包含String
     * @return boolean
     */
    public static boolean isBasic(Object obj, boolean includeString) {
        return obj instanceof Boolean || obj instanceof Number || (includeString && obj instanceof String);
    }

    public static Type[] getActualType(Field field) {
        Class<?> type = field.getType();
        if (Arrays.asList(Map.class, Set.class, List.class).contains(type)) {
            return ((ParameterizedType) field.getGenericType()).getActualTypeArguments();
        } else {
            return new Type[0];
        }
    }

    /**
     * 获取方法签名
     *
     * @param method 方法
     * @return String
     */
    public static String getMethodSignature(Method method) {
        return getMethodSignature(method, true);
    }

    /**
     * 获取方法签名
     *
     * @param method        方法
     * @param withParamInfo 是否复贷参数信息
     * @return String
     */
    public static String getMethodSignature(Method method, boolean withParamInfo) {
        String clzName = method.getDeclaringClass().getName();
        String methodName = method.getName();
        StringBuilder params = new StringBuilder();
        if (withParamInfo) {
            Parameter[] parameters = method.getParameters();
            params.append("(");
            Arrays.stream(parameters).forEach(parameter -> params.append(parameter.getType().getName()).append(","));
            params.deleteCharAt(params.length() - 1);
            params.append(")");
        }
        return clzName + "." + methodName + params;
    }

    /**
     * 设值对象属性值
     *
     * @param obj       对象
     * @param fieldName 属性
     * @param value     值
     */
    public static void setProperty(Object obj, String fieldName, Object value) {
        try {
            Class<?> clz = obj.getClass();
            Field field = clz.getDeclaredField(fieldName);
            field.setAccessible(true);
            field.set(obj, value);
        } catch (NoSuchFieldException | IllegalAccessException e) {
            throw new BizException("设值属性值失败", e);
        }
    }

    public static Object getProperty(Object obj, String fieldName) {
        try {
            Class<?> clz = obj.getClass();
            Field field = clz.getDeclaredField(fieldName);
            field.setAccessible(true);
            return field.get(obj);
        } catch (NoSuchFieldException | IllegalAccessException e) {
            throw new BizException("设值属性值失败", e);
        }
    }
}
