package cn.wjee.commons.lang;

import cn.wjee.commons.constants.Vars;
import cn.wjee.commons.exception.BizException;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * 正则工具
 *
 * @author lxn
 */
public class RegExpUtils {
    private RegExpUtils() {
    }

    /**
     * Check Matches
     *
     * @param pattern 类型
     * @param value   值
     * @return boolean
     */
    public static boolean matches(String pattern, String value) {
        return !StringUtils.isAnyBlank(pattern, value) && Pattern.matches(pattern, value);
    }

    /**
     * 是否手机号码
     *
     * @param value 值
     * @return boolean
     */
    public static boolean mobile(String value) {
        return StringUtils.isNotBlank(value) && Pattern.matches("^1[0-9]{10}$", value);
    }

    /**
     * 是否合法日期
     *
     * @param value       值
     * @param yearSepKey  - 年后分割字符
     * @param monthSepKey - 月后分割字符
     * @param daySepKey   - 日后分割字符
     * @return boolean
     */
    public static boolean isDate(String value, String yearSepKey, String monthSepKey, String daySepKey) {
        if (StringUtils.isBlank(value)) {
            return false;
        }
        StringBuilder pattern = new StringBuilder();
        pattern.append("^((?:(?!0000)[0-9]{4}(").append(yearSepKey);
        pattern.append(")(?:(?:0[1-9]|1[0-2])(").append(monthSepKey);
        pattern.append(")(?:0[1-9]|1[0-9]|2[0-8])|(?:0[13-9]|1[0-2])(").append(monthSepKey);
        pattern.append(")(?:29|30)|(?:0[13578]|1[02])(").append(monthSepKey);
        pattern.append(")31)|(?:[0-9]{2}(?:0[48]|[2468][048]|[13579][26])|");
        pattern.append("(?:0[48]|[2468][048]|[13579][26])00)(").append(monthSepKey);
        pattern.append(")02(").append(monthSepKey).append(")29))").append(daySepKey).append("$");
        return Pattern.matches(pattern.toString(), value);
    }

    /**
     * 是否合法日期 yyyy-MM-dd
     *
     * @param value 值
     * @return boolean
     */
    public static boolean isDateYmd(String value) {
        return isDate(value, "-", "-", "");
    }

    /**
     * 是否为数字
     *
     * @param value 值
     * @return boolean
     */
    public static boolean isNumeric(String value) {
        if (StringUtils.isBlank(value)) {
            return false;
        }
        if (StringUtils.equals(Vars.HYPHEN, value)) {
            return false;
        }
        final int sz = value.length();
        for (int i = 0; i < sz; i++) {
            char character = value.charAt(i);
            boolean isSign = Vars.HYPHEN.equals(String.valueOf(character));
            boolean isDigit = Character.isDigit(character);
            if (i == 0 && !isSign && !isDigit) {
                return false;
            }
            if (i > 0 && !isDigit) {
                return false;
            }
        }
        return true;
    }

    /**
     * 是否Decimal
     *
     * @param value     值
     * @param precision 整数
     * @param scale     小数
     * @return boolean
     */
    public static boolean isDecimal(String value, int precision, int scale) {
        if (precision <= 0 || scale <= 0 || precision - scale - 1 < 0) {
            throw new BizException("invalid presion scale param");
        }
        String pattern = "^(([1-9]\\d{0," + (precision - scale - 1) + "})|0)(\\.\\d{0," + scale + "})?$";
        return Pattern.matches(pattern, value);
    }

    /**
     * Validate is decimal , has no length limit
     *
     * @param value 值
     * @return boolean
     */
    public static boolean isWildDecimal(String value) {
        String pattern = "^-?(([1-9]\\d{0,})|0)(\\.\\d{0,})?$";
        return Pattern.matches(pattern, value);
    }

    /**
     * 是否汉字
     *
     * @param value 值
     * @return boolean
     */
    public static boolean isChineseCharacter(String value) {
        return Pattern.matches("[\u4e00-\u9fa5]*", value);
    }


    /**
     * Check IS IP
     *
     * @param value 值
     * @return boolean
     */
    public static boolean isIp(String value) {
        String pattern = "^(\\d{1,2}|1\\d\\d|2[0-4]\\d|25[0-5])((\\.(\\d{1,2}|1\\d\\d|2[0-4]\\d|25[0-5])){3}|(\\.(\\d{1,2}|1\\d\\d|2[0-4]\\d|25[0-5])){5})$";
        return matches(pattern, value);
    }

    /**
     * 获取URL中参数变量的值
     *
     * @param url  Get参数链接
     * @param name 参数名
     * @return 返回值
     */
    public static String parseUrlParam(String url, String name) {
        String pattern = "(^|&|\\?)" + name + "=([^&]*)(&|$)";
        Pattern compilePattern = Pattern.compile(pattern);
        Matcher matcher = compilePattern.matcher(url);
        String result = "";
        while (matcher.find()) {
            result = matcher.group(2);
        }
        return result;
    }

    /**
     * 截取HTML中简单标签
     *
     * @param h5      Html内容
     * @param tagName 标签名
     * @return List
     */
    public static List<String> findH5SimpleTags(String h5, String tagName) {
        Pattern pattern = Pattern.compile("<" + tagName + "\\b.*?(?:>|/>)", Pattern.CASE_INSENSITIVE);
        Matcher matcher = pattern.matcher(h5);
        List<String> result = new ArrayList<>();
        while (matcher.find()) {
            result.add(matcher.group());
        }
        return result;
    }

    /**
     * 截取HTML中简单标签属性值
     *
     * @param h5      Html内容
     * @param tagName 标签
     * @param attr    属性
     * @return List
     */
    public static List<String> findH5SimpleTagAttr(String h5, String tagName, String attr) {
        return RegExpUtils.findH5SimpleTags(h5, tagName).stream().map(item -> {
            Pattern pattern = Pattern.compile("(?<=" + attr + "=['\"]).*?(?=['\"])", Pattern.CASE_INSENSITIVE);
            Matcher matcher = pattern.matcher(item);
            String mapSrc = "";
            while (matcher.find()) {
                mapSrc = matcher.group();
            }
            return mapSrc;
        }).collect(Collectors.toList());
    }

    /**
     * 获取H5图片标签链接
     *
     * @param h5 HTML
     * @return List
     */
    public static List<String> findH5ImgSrc(String h5) {
        return RegExpUtils.findH5SimpleTagAttr(h5, "img", "src");
    }

    /**
     * 检测连接是否属于某个域名
     *
     * @param url       连接
     * @param topDomain 域名
     * @return boolean
     */
    public static boolean isOfDomain(String url, String topDomain) {
        try {
            if (StringUtils.isAnyBlank(url, topDomain)) {
                return false;
            }

            final String domain = StringUtils.removeLastChar(topDomain, "/");
            return Pattern.compile("^http(s)?://([\\.\\da-zA-Z_-]+\\.)" + domain + "/").matcher(url).find();
        } catch (Exception e) {
            return false;
        }
    }

    /**
     * 局域网可使用的网段（私网地址段）有三大段：
     * <p>
     * 10.0.0.0~10.255.255.255（A类）
     * 172.16.0.0~172.31.255.255（B类）
     * 192.168.0.0~192.168.255.255（C类）
     *
     * @param ip IP
     * @return boolean
     */
    public static boolean isInnerIp(String ip) {
        if (StringUtils.isBlank(ip)) {
            return false;
        }
        String trimIp = ip.trim();
        String patternA = "^10\\.(1\\d{2}|2[0-4]\\d|25[0-5]|[1-9]\\d|[0-9])\\.(1\\d{2}|2[0-4]\\d|25[0-5]|[1-9]\\d|[0-9])\\.(1\\d{2}|2[0-4]\\d|25[0-5]|[1-9]\\d|[0-9])$";
        if (matches(patternA, trimIp)) {
            return true;
        }
        String patternB = "^172\\.(1[6789]|2[0-9]|3[01])\\.(1\\d{2}|2[0-4]\\d|25[0-5]|[1-9]\\d|[0-9])\\.(1\\d{2}|2[0-4]\\d|25[0-5]|[1-9]\\d|[0-9])$";
        if (matches(patternB, trimIp)) {
            return true;
        }
        String patternC = "^192\\.168\\.(1\\d{2}|2[0-4]\\d|25[0-5]|[1-9]\\d|[0-9])\\.(1\\d{2}|2[0-4]\\d|25[0-5]|[1-9]\\d|[0-9])$";
        if (matches(patternC, trimIp)) {
            return true;
        }
        return Vars.LOCAL_HOST.equalsIgnoreCase(trimIp) || Vars.LOCAL_IP.equals(trimIp);
    }

}
