package cn.wjee.commons.lang;


import cn.wjee.commons.collection.MapUtils;
import cn.wjee.commons.constants.Vars;

import java.net.URL;
import java.text.MessageFormat;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;

/**
 * 字符串工具类
 *
 * @author listening
 */
public class StringUtils {
    /**
     * 换行
     */
    public static final String WRAP_LINE = Vars.WRAP;

    private StringUtils() {

    }

    /**
     * 是否非空
     *
     * @param value 值
     * @return Boolean
     */
    public static boolean isBlank(String value) {
        return value == null || value.trim().isEmpty();
    }

    /**
     * 是否非空
     *
     * @param value 值
     * @return Boolean
     */
    public static boolean isNotBlank(String value) {
        return !isBlank(value);
    }

    /**
     * 去首位空格
     *
     * @param value 值
     * @return String
     */
    public static String trim(String value) {
        return value != null ? value.trim() : "";
    }

    /**
     * 首字母大写
     *
     * @param value 字符串
     * @return String
     */
    public static String capitalize(String value) {
        if (StringUtils.isBlank(value)) {
            return value;
        }
        String tempValue = value.trim();
        return tempValue.substring(0, 1).toUpperCase() + tempValue.substring(1);
    }

    /**
     * 首字母小写
     *
     * @param value 字符串
     * @return String
     */
    public static String unCapitalize(String value) {
        if (StringUtils.isBlank(value)) {
            return value;
        }
        String tempValue = value.trim();
        return tempValue.substring(0, 1).toLowerCase() + tempValue.substring(1);
    }

    /**
     * 转小写
     *
     * @param value 字符串
     * @return String
     */
    public static String lowerCase(String value) {
        String tempValue = getValue(value);
        return StringUtils.isBlank(tempValue) ? tempValue : tempValue.toLowerCase();
    }

    /**
     * 转大写
     *
     * @param value 字符串
     * @return String
     */
    public static String upperCase(String value) {
        String tempValue = getValue(value);
        return StringUtils.isBlank(tempValue) ? tempValue : tempValue.toUpperCase();
    }

    /**
     * 比较字符串相等
     *
     * @param valueOne 字符串
     * @param valueTwo 比较字符串
     * @return boolean
     */
    public static boolean equals(String valueOne, String valueTwo) {
        return !StringUtils.isAnyBlank(valueOne, valueTwo) && valueOne.equals(valueTwo);
    }

    /**
     * 比较字符串相等(忽略大小写)
     *
     * @param valueOne 字符串
     * @param valueTwo 比较字符串
     * @return boolean
     */
    public static boolean equalsIgnoreCase(String valueOne, String valueTwo) {
        return !StringUtils.isAnyBlank(valueOne, valueTwo) && valueOne.equalsIgnoreCase(valueTwo);
    }

    /**
     * Trim Blank Characters Or Wraps
     *
     * @param value 值
     * @return String
     */
    public static String trimWrap(String value) {
        String tempValue = value == null ? "" : value.trim();
        return tempValue.replaceAll("\\s+", "");
    }

    /**
     * IS All Blank
     *
     * @param values 值
     * @return boolean
     */
    public static boolean isAllBlank(String... values) {
        return Arrays.stream(values).noneMatch(StringUtils::isNotBlank);
    }

    /**
     * IS Any Blank
     *
     * @param values 值
     * @return boolean
     */
    public static boolean isAnyBlank(String... values) {
        return Arrays.stream(values).anyMatch(StringUtils::isBlank);
    }

    /**
     * 字符长度
     *
     * @param value 值
     * @return int
     */
    public static int length(String value) {
        return StringUtils.isBlank(value) ? 0 : value.trim().length();
    }

    /**
     * 字符长度区间
     *
     * @param value 值
     * @param min   最小
     * @param max   最大
     * @return boolean
     */
    public static boolean length(String value, int min, int max) {
        int length = StringUtils.isBlank(value) ? 0 : value.trim().length();
        return length >= min && length <= max;
    }

    /**
     * 字符长度区间
     *
     * @param value 值
     * @param min   最小
     * @param max   最大
     * @return boolean
     */
    public static boolean bytesLength(String value, int min, int max) {
        int length = bytesLength(value);
        return length >= min && length <= max;
    }

    /**
     * get the string value bytes length
     *
     * @param value 值
     * @return int
     */
    public static int bytesLength(String value) {
        if (StringUtils.isBlank(value)) {
            return 0;
        }
        String content = value.replaceAll("[^\\x00-\\xff]", "**");
        return content.length();
    }

    /**
     * 格式化字符串
     *
     * @param value  值
     * @param params 参数
     * @return String
     */
    public static String format(String value, Object... params) {
        return MessageFormat.format(value, params);
    }

    /**
     * 判断两个值是否相同
     *
     * @param valueOne 值1
     * @param valueTwo 值2
     * @return boolean
     */
    public static boolean isSame(String valueOne, String valueTwo) {
        // 都是空则相同(忽略空串和null,默认统一处理)
        if (isBlank(valueOne) && isBlank(valueTwo)) {
            return true;
        }
        // 都不为空,equal比较
        if (isNotBlank(valueOne) && isNotBlank(valueTwo)) {
            return valueOne.trim().equals(valueTwo.trim());
        }
        // 其一为空返回false
        return false;
    }

    /**
     * 处理String值非空去空格否则返回空串
     *
     * @param value 值
     * @return String
     */
    public static String getValue(String value) {
        return getValue(value, "");
    }

    /**
     * 处理String值非空去空格否则返回默认值
     *
     * @param value 值
     * @param def   默认值
     * @return String
     */
    public static String getValue(String value, String def) {
        return getValue(value, def, null);
    }

    /**
     * 处理String值非空去空格否则返回默认值
     *
     * @param value  值
     * @param def    默认值
     * @param maxLen 返回最大长度
     * @return String
     */
    public static String getValue(String value, String def, Integer maxLen) {
        if (isBlank(value)) {
            return def;
        }
        String trimValue = value.trim();
        if (maxLen != null && maxLen > 0 && maxLen < trimValue.length()) {
            return trimValue.substring(0, maxLen);
        }
        return trimValue;
    }

    /**
     * Int转String
     *
     * @param value 参数值
     * @param def   默认值
     * @return String
     */
    public static String getValue(Integer value, Integer def) {
        String defaultValue = def != null ? String.valueOf(def) : null;
        return (value == null) ? defaultValue : String.valueOf(value);
    }

    /**
     * Parse String To Integer
     *
     * @param value 待转换值
     * @return Integer
     */
    public static Integer getIntValue(String value) {
        return getIntValue(value, null);
    }

    /**
     * Parse String To Integer
     *
     * @param value 待转换值
     * @param def   默认值
     * @return Integer
     */
    public static Integer getIntValue(String value, Integer def) {
        if (isBlank(value)) {
            return def;
        }
        String trimVal = getValue(value);
        if (!StringUtils.isNumeric(trimVal)) {
            return def;
        }
        return Integer.parseInt(value);
    }

    /**
     * Parse String To Long
     *
     * @param value 待转换值
     * @return Long
     */
    public static Long getLongValue(String value) {
        return getLongValue(value, null);
    }

    /**
     * Parse String To Long
     *
     * @param value 待转换值
     * @param def   默认值
     * @return Long
     */
    public static Long getLongValue(String value, Long def) {
        if (isBlank(value)) {
            return def;
        }
        String trimVal = getValue(value);
        if (!RegExpUtils.isNumeric(trimVal)) {
            return def;
        }
        return Long.valueOf(trimVal);
    }

    /**
     * 过滤emoji 或者 其他非文字类型的字符
     *
     * @param content 字符串
     * @return String
     */
    public static String filterEmoji(String content) {
        if (StringUtils.isBlank(content)) {
            return "";
        }
        StringBuilder sb = new StringBuilder();
        for (char ch : content.toCharArray()) {
            if (!Character.isHighSurrogate(ch) && !Character.isLowSurrogate(ch)) {
                sb.append(ch);
            }
        }
        return sb.toString();
    }

    /**
     * 生成Get方法名称
     *
     * @param fieldName 字段名
     * @return String
     */
    public static String getGetterMethodName(String fieldName) {
        return "get" + capitalize(fieldName);
    }

    /**
     * 生成Set方法名称
     *
     * @param fieldName 字段名
     * @return String
     */
    public static String getSetterMethodName(String fieldName) {
        return "set" + capitalize(fieldName);
    }

    /**
     * 包名转路径
     *
     * @param packageName 字段名
     * @return String
     */
    public static String getPackagePath(String packageName) {
        return StringUtils.isBlank(packageName) ? "" : packageName.replace(".", "/");
    }

    /**
     * 字符串脱敏
     *
     * @param value   字符串
     * @param start   开始位
     * @param end     结束位
     * @param replace 替换成指定字符
     * @return String
     */
    public static String safeString(String value, int start, int end, char replace) {
        if (StringUtils.isBlank(value)) {
            return value;
        }
        if (value.length() <= 1) {
            return value;
        }
        char[] charArray = value.toCharArray();
        if (value.length() == 2) {
            charArray[1] = replace;
            return String.valueOf(charArray);
        }
        end = Math.min(end, charArray.length);
        for (int i = 0; i < charArray.length; i++) {
            if (i >= start - 1 && i <= end - 1) {
                charArray[i] = replace;
            }
        }
        return String.valueOf(charArray);
    }

    /**
     * 通用脱敏
     *
     * @param value 字符串
     * @return String
     */
    public static String safeValue(String value) {
        if (StringUtils.isBlank(value)) {
            return "";
        }
        String tempValue = StringUtils.trim(value);
        if (StringUtils.isBlank(tempValue) || StringUtils.length(tempValue) <= 1) {
            return tempValue;
        }
        if (StringUtils.length(tempValue) == 2) {
            return tempValue.charAt(0) + "*";
        }
        return safeString(value, 2, tempValue.length() - 1, '*');
    }

    /**
     * 平均脱敏
     *
     * @param value 值
     * @return 脱敏后的值
     */
    public static String safeValueAvg(String value) {
        if (StringUtils.isBlank(value)) {
            return "";
        }
        String tempValue = StringUtils.trim(value);
        int segSize = tempValue.length() / 3;
        if (segSize < 1) {
            return safeValue(tempValue);
        }
        return safeString(value, segSize + 1, value.length() - segSize, '*');
    }

    /**
     * 手机号脱敏
     *
     * @param value 手机号
     * @return String
     */
    public static String safeMobile(String value) {
        return safeString(value, 4, 7, '*');
    }

    /**
     * 字符串尾部截取指定个数字符
     *
     * @param value 字符串
     * @param count 字符数
     * @return String
     */
    public static String lastChars(String value, int count) {
        if (StringUtils.isBlank(value) || count <= 0) {
            return "";
        }
        if (value.length() < count) {
            return value;
        }
        return value.substring(value.length() - count);
    }

    /**
     * 驼峰命名转中横线分割
     *
     * @param value String
     * @return String
     */
    public static String camel2line(String value) {
        String tempValue = isBlank(value) ? "" : value;
        return lowerCase(unCapitalize(tempValue).replaceAll("[A-Z]", "-$0"));
    }

    /**
     * 是否以指定字符串开头
     *
     * @param value 字符串
     * @param s     匹配串
     * @return boolean
     */
    public static boolean startsWith(String value, String s) {
        if (StringUtils.isAnyBlank(value, s)) {
            return false;
        }
        return value.startsWith(s);
    }

    /**
     * 是否以指定字符串结尾
     *
     * @param value 字符串
     * @param s     匹配串
     * @return boolean
     */
    public static boolean endsWith(String value, String s) {
        if (StringUtils.isAnyBlank(value, s)) {
            return false;
        }
        return value.endsWith(s);
    }

    /**
     * 截取字符串
     *
     * @param str   字符串
     * @param start 开始位置
     * @param end   结束位置
     * @return String
     */
    public static String substring(String str, int start, int end) {
        if (StringUtils.isBlank(str) || start < 0 || end < 0 || start > end) {
            return "";
        }
        return str.substring(start, end);
    }

    /**
     * 字符串截取
     *
     * @param str   字符串
     * @param start 开始位
     * @return String
     */
    public static String substring(String str, int start) {
        return substring(str, start, str.length());
    }

    /**
     * 截取最大长度字符
     *
     * @param str    字符串
     * @param maxLen 截取最大字符数
     * @return String
     */
    public static String maxString(String str, int maxLen) {
        if (StringUtils.isBlank(str) || maxLen < 0) {
            return "";
        }
        return str.substring(0, Math.min(maxLen, str.length()));
    }

    /**
     * 字符串拼接
     *
     * @param values    字符串数组
     * @param separator 分割符
     * @return String
     */
    public static String join(String[] values, String separator) {
        if (values == null || values.length == 0) {
            return "";
        }
        return String.join(separator, values);
    }

    /**
     * 字符串拼接
     *
     * @param list      字符串列表
     * @param separator 分割符
     * @return String
     */
    public static String join(List<String> list, String separator) {
        if (list == null || list.isEmpty()) {
            return "";
        }
        return join(list.toArray(new String[0]), separator);
    }

    /**
     * 字符串切分
     *
     * @param str       字符串
     * @param separator 分隔符
     * @return String[]
     */
    public static String[] split(String str, String separator) {
        if (StringUtils.isBlank(str) || StringUtils.isBlank(separator)) {
            return new String[]{str};
        }
        return str.split(separator);
    }

    /**
     * 是否为数字
     *
     * @param str 字符串
     * @return boolean
     */
    public static boolean isNumeric(String str) {
        return RegExpUtils.isNumeric(str);
    }

    /**
     * 是否包含字符子串
     *
     * @param value  字符串
     * @param search 检索字符
     * @return boolean
     */
    public static boolean contains(String value, String search) {
        return !StringUtils.isAnyBlank(value, search) && value.contains(search);
    }

    /**
     * Column Field To Java
     *
     * @param fieldName    字段名
     * @param isFirstLower 是否首字母小写
     * @return String
     */
    public static String columnFieldToJava(String fieldName, boolean isFirstLower) {
        if (StringUtils.isBlank(fieldName)) {
            return null;
        }
        StringTokenizer tokenizer = new StringTokenizer(fieldName, "_");
        int index = 0;
        StringBuilder result = new StringBuilder();
        while (tokenizer.hasMoreTokens()) {
            String value = tokenizer.nextToken();
            if (index == 0 && isFirstLower) {
                result.append(value.toLowerCase());
            } else {
                result.append(StringUtils.capitalize(value.toLowerCase()));
            }
            index++;
        }
        return result.toString();
    }

    /**
     * 左边补符号
     *
     * @param value     字符串
     * @param padLength 补充长度
     * @param padChar   补充字符
     * @return String
     */
    public static String lpad(String value, int padLength, String padChar) {
        String tempValue = StringUtils.isBlank(value) ? "" : value;
        if (padLength <= 0) {
            return tempValue;
        }
        StringBuilder builder = new StringBuilder();
        for (int i = 0; i < padLength; i++) {
            builder.append(padChar);
        }
        builder.append(tempValue);
        return builder.toString();
    }

    /**
     * 右边补符号
     *
     * @param value     字符串
     * @param padLength 补充长度
     * @param padChar   补充字符
     * @return String
     */
    public static String rpad(String value, int padLength, String padChar) {
        String tempValue = StringUtils.isBlank(value) ? "" : value;
        if (padLength <= 0) {
            return tempValue;
        }
        StringBuilder builder = new StringBuilder(tempValue);
        for (int i = 0; i < padLength; i++) {
            builder.append(padChar);
        }
        return builder.toString();
    }

    /**
     * 左补空格
     *
     * @param value 字符串
     * @param size  补空格数
     * @return String
     */
    public static String lpadBlank(String value, int size) {
        return lpad(value, size, " ");
    }

    /**
     * 模板参数填充， 如将模板中的变量：${name} 替换为参数Map中的值
     *
     * @param template    模板内容
     * @param paramPrefix 参数前缀，如: ${
     * @param paramSuffix 参数后缀，如:}
     * @param paramMap    填充参数Map
     * @return String
     */
    public static String format(String template, String paramPrefix, String paramSuffix, Map<String, String> paramMap) {
        if (StringUtils.isBlank(template)) {
            return "";
        }
        if (MapUtils.isEmpty(paramMap)) {
            return template;
        }
        paramPrefix = getValue(paramPrefix);
        paramSuffix = getValue(paramSuffix);
        for (Map.Entry<String, String> entry : paramMap.entrySet()) {
            template = template.replaceAll(paramPrefix + entry.getKey() + paramSuffix, entry.getValue());
        }
        return template;
    }

    /**
     * 是否是Http链接
     *
     * @param value 地址
     * @return boolean
     */
    public static boolean isHttp(String value) {
        try {
            if (StringUtils.isBlank(value)) {
                return false;
            }
            new URL(StringUtils.trim(value));
            return true;
        } catch (Exception e) {
            return false;
        }
    }

    /**
     * 去除尾部指定字符串，直到没有
     *
     * @param value 原串
     * @param chars 替换串
     * @return String
     */
    public static String removeLastChar(String value, String chars) {
        String tempValue = getValue(value);
        if (StringUtils.isBlank(value) || StringUtils.isBlank(chars)) {
            return "";
        }
        while (StringUtils.endsWith(tempValue, chars)) {
            tempValue = StringUtils.substring(tempValue, 0, tempValue.length() - chars.length());
        }
        return tempValue;
    }

    /**
     * 身份证脱敏
     *
     * @param value 身份证号
     * @return String
     */
    public static String safeIdentity(String value) {
        if (StringUtils.isBlank(value)) {
            return "";
        }
        if (StringUtils.length(value) == 15) {
            return safeString(value, 7, 12, '*');
        }
        if (StringUtils.length(value) == 18) {
            return safeString(value, 7, 14, '*');
        }
        return value;
    }


    /**
     * 邮箱脱敏(@符号前: 保留首尾字母，@符号后: 保持不变)
     *
     * @param value 手机号
     * @return String
     */
    public static String safeEmail(String value) {
        final String delimiter = "@";
        if (StringUtils.isBlank(value) || !StringUtils.contains(value, delimiter)) {
            return value;
        }
        int indexOfDelimiter = value.indexOf(delimiter);
        String segmentFirst = StringUtils.substring(value, 0, indexOfDelimiter);
        String segmentLast = StringUtils.substring(value, indexOfDelimiter);
        return safeValue(segmentFirst) + segmentLast;
    }

    /**
     * 重复拼接
     *
     * @param value 字符
     * @param times 次数
     * @return String
     */
    public static String repeat(CharSequence value, int times) {
        StringBuilder buffer = new StringBuilder();
        for (int i = 0; times > 0 && i < times; i++) {
            buffer.append(value);
        }
        return buffer.toString();
    }
}
