package cn.wjee.commons.lang.html;

import cn.wjee.commons.exception.Asserts;

/**
 * Html相关转义工具类(From Spring)。
 */
public abstract class HtmlUtils {

    /**
     * Shared instance of pre-parsed HTML character entity references.
     */
    private static final HtmlCharacterEntityReferences characterEntityReferences = new HtmlCharacterEntityReferences();

    /**
     * Turn special characters into HTML character references.
     * <p>Handles complete character set defined in HTML 4.01 recommendation.
     * <p>Escapes all special characters to their corresponding
     * entity reference (e.g. {@code &lt;}).
     * <p>Reference:
     * <a href="https://www.w3.org/TR/html4/sgml/entities.html">
     * https://www.w3.org/TR/html4/sgml/entities.html
     * </a>
     *
     * @param input the (unescaped) input string
     * @return the escaped string
     */
    public static String htmlEscape(String input) {
        return htmlEscape(input, HtmlCharacterEntityReferences.DEFAULT_CHARACTER_ENCODING);
    }

    /**
     * Turn special characters into HTML character references.
     * <p>Handles complete character set defined in HTML 4.01 recommendation.
     * <p>Escapes all special characters to their corresponding
     * entity reference (e.g. {@code &lt;}) at least as required by the
     * specified encoding. In other words, if a special character does
     * not have to be escaped for the given encoding, it may not be.
     * <p>Reference:
     * <a href="https://www.w3.org/TR/html4/sgml/entities.html">
     * https://www.w3.org/TR/html4/sgml/entities.html
     * </a>
     *
     * @param input    the (unescaped) input string
     * @param encoding the name of a supported {@link java.nio.charset.Charset charset}
     * @return the escaped string
     * @since 4.1.2
     */
    public static String htmlEscape(String input, String encoding) {
        Asserts.notNull(input, "Input is required");
        Asserts.notNull(encoding, "Encoding is required");
        StringBuilder escaped = new StringBuilder(input.length() * 2);
        for (int i = 0; i < input.length(); i++) {
            char character = input.charAt(i);
            String reference = characterEntityReferences.convertToReference(character, encoding);
            if (reference != null) {
                escaped.append(reference);
            } else {
                escaped.append(character);
            }
        }
        return escaped.toString();
    }

    /**
     * Turn special characters into HTML character references.
     * <p>Handles complete character set defined in HTML 4.01 recommendation.
     * <p>Escapes all special characters to their corresponding numeric
     * reference in decimal format
     * <p>Reference:
     * <a href="https://www.w3.org/TR/html4/sgml/entities.html">
     * https://www.w3.org/TR/html4/sgml/entities.html
     * </a>
     *
     * @param input the (unescaped) input string
     * @return the escaped string
     */
    public static String htmlEscapeDecimal(String input) {
        return htmlEscapeDecimal(input, HtmlCharacterEntityReferences.DEFAULT_CHARACTER_ENCODING);
    }

    /**
     * Turn special characters into HTML character references.
     * <p>Handles complete character set defined in HTML 4.01 recommendation.
     * <p>Escapes all special characters to their corresponding numeric
     * reference in decimal format at least as required by the
     * specified encoding. In other words, if a special character does
     * not have to be escaped for the given encoding, it may not be.
     * <p>Reference:
     * <a href="https://www.w3.org/TR/html4/sgml/entities.html">
     * https://www.w3.org/TR/html4/sgml/entities.html
     * </a>
     *
     * @param input    the (unescaped) input string
     * @param encoding the name of a supported {@link java.nio.charset.Charset charset}
     * @return the escaped string
     * @since 4.1.2
     */
    public static String htmlEscapeDecimal(String input, String encoding) {
        Asserts.notNull(input, "Input is required");
        Asserts.notNull(encoding, "Encoding is required");
        StringBuilder escaped = new StringBuilder(input.length() * 2);
        for (int i = 0; i < input.length(); i++) {
            char character = input.charAt(i);
            if (characterEntityReferences.isMappedToReference(character, encoding)) {
                escaped.append(HtmlCharacterEntityReferences.DECIMAL_REFERENCE_START);
                escaped.append((int) character);
                escaped.append(HtmlCharacterEntityReferences.REFERENCE_END);
            } else {
                escaped.append(character);
            }
        }
        return escaped.toString();
    }

    /**
     * Turn special characters into HTML character references.
     * <p>Handles complete character set defined in HTML 4.01 recommendation.
     * <p>Escapes all special characters to their corresponding numeric
     * reference in hex format
     * <p>Reference:
     * <a href="https://www.w3.org/TR/html4/sgml/entities.html">
     * https://www.w3.org/TR/html4/sgml/entities.html
     * </a>
     *
     * @param input the (unescaped) input string
     * @return the escaped string
     */
    public static String htmlEscapeHex(String input) {
        return htmlEscapeHex(input, HtmlCharacterEntityReferences.DEFAULT_CHARACTER_ENCODING);
    }

    /**
     * Turn special characters into HTML character references.
     * <p>Handles complete character set defined in HTML 4.01 recommendation.
     * <p>Escapes all special characters to their corresponding numeric
     * reference in hex format at least as required by the
     * specified encoding. In other words, if a special character does
     * not have to be escaped for the given encoding, it may not be.
     * <p>Reference:
     * <a href="https://www.w3.org/TR/html4/sgml/entities.html">
     * https://www.w3.org/TR/html4/sgml/entities.html
     * </a>
     *
     * @param input    the (unescaped) input string
     * @param encoding the name of a supported {@link java.nio.charset.Charset charset}
     * @return the escaped string
     * @since 4.1.2
     */
    public static String htmlEscapeHex(String input, String encoding) {
        Asserts.notNull(input, "Input is required");
        Asserts.notNull(encoding, "Encoding is required");
        StringBuilder escaped = new StringBuilder(input.length() * 2);
        for (int i = 0; i < input.length(); i++) {
            char character = input.charAt(i);
            if (characterEntityReferences.isMappedToReference(character, encoding)) {
                escaped.append(HtmlCharacterEntityReferences.HEX_REFERENCE_START);
                escaped.append(Integer.toString(character, 16));
                escaped.append(HtmlCharacterEntityReferences.REFERENCE_END);
            } else {
                escaped.append(character);
            }
        }
        return escaped.toString();
    }

    /**
     * Turn HTML character references into their plain text UNICODE equivalent.
     * <p>Handles complete character set defined in HTML 4.01 recommendation
     * and all reference types (decimal, hex, and entity).
     * <p>Correctly converts the following formats:
     * <blockquote>
     * &amp;#<i>Entity</i>; - <i>(Example: &amp;amp;) case sensitive</i>
     * &amp;#<i>Decimal</i>; - <i>(Example: &amp;#68;)</i><br>
     * &amp;#x<i>Hex</i>; - <i>(Example: &amp;#xE5;) case insensitive</i><br>
     * </blockquote>
     * <p>Gracefully handles malformed character references by copying original
     * characters as is when encountered.
     * <p>Reference:
     * <a href="https://www.w3.org/TR/html4/sgml/entities.html">
     * https://www.w3.org/TR/html4/sgml/entities.html
     * </a>
     *
     * @param input the (escaped) input string
     * @return the unescaped string
     */
    public static String htmlUnescape(String input) {
        return new HtmlCharacterEntityDecoder(characterEntityReferences, input).decode();
    }

}
